/*
 *  Copyright (C)  Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


package de.lunqual.rzpro.database;

// eigene Pakete
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

import javax.swing.JComboBox;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.fenster.dialoge.DialogControl;
import de.lunqual.rzpro.items.currency.CurrencyItem;
import de.lunqual.rzpro.items.currency.CurrencyListe;
import de.lunqual.rzpro.items.dialog.DialogItem;
import de.lunqual.rzpro.log.LogFactory;
/**
 *
 * @author  lunqual
 */
public class DBCurrency {

    RzPro           				rz;
    Connection      			con;
    DBFactory       			db;

    PreparedStatement   	getCurrencyListe;
    PreparedStatement   	getCurrency;
    PreparedStatement   	saveCurrency;
    PreparedStatement   	updateCurrency;
    PreparedStatement   	deleteCurrency;
    PreparedStatement   	checkCurrencyUsed;

    CurrencyItem		   	euro;
    DialogItem					dlgItem;
    /** Creates a new instance of DBCurrency */
    public DBCurrency(RzPro r, DBFactory aDb, Connection aCon) {
        rz = r;
        db = aDb;
        con = aCon;
        setStatements();
        dlgItem = new DialogItem(0,"",0.0,"","","","","","",null);
        euro = this.dbGetCurrency(1,true);
    }

    private void setStatements(){
        try{
            getCurrencyListe    = con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_CURRENCIES + " ORDER BY shortname",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getCurrency         = con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_CURRENCIES + " where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            updateCurrency      = con.prepareStatement("UPDATE " + DBFactory.TABLE_CURRENCIES + " SET name=?,shortname=?,rate=?,datum=?,user_1=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            saveCurrency        = con.prepareStatement("INSERT INTO " + DBFactory.TABLE_CURRENCIES + " (name,shortname,rate,datum,user_1) VALUES(?,?,?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            deleteCurrency      = con.prepareStatement("DELETE FROM " + DBFactory.TABLE_CURRENCIES + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            checkCurrencyUsed   = con.prepareStatement("SELECT waehrung FROM " + DBFactory.TABLE_REZEPTLISTE + " where waehrung=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBCurrency.setStatements", e.getLocalizedMessage());
        }
    }

    /** boolean dbCheckCurrencyUsed(int) pr�ft ob eine W�hrung in der
     * Rezeptliste verwendet wird
     * @@return false wenn nicht verwendet, true wenn verwendet
     */
    public boolean dbCheckCurrencyUsed(int aID){
        boolean ret = false;
        try{
            checkCurrencyUsed.setInt(1, aID);
            checkCurrencyUsed.executeQuery();
            final ResultSet rs = checkCurrencyUsed.getResultSet();
            if(rs.next()){
                ret = true;
            }
            if(rs != null) {
				rs.close();
			}
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBCurrency.dbDeleteCurrency:", e.getLocalizedMessage());
            ret = true;
        }
        return ret;
    }

    /** void dbDeleteCurrency(int) l�scht eine W�hrung wenn diese nicht
     * verwendet wird...
     */

    public void dbDeleteCurrency(int aID,String titel){
        try{
            if (aID != 1){ // Default darf nicht gelöscht werden !!!
                if (!dbCheckCurrencyUsed(aID)){ // wenn nicht in der Rezeptliste verwendet
                    deleteCurrency.setInt(1, aID);
                    if(deleteCurrency.executeUpdate() == 0){
                        dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
                        dlgItem.setCaption(rz.getLocale().getString("waehrung.key_not_found"));
                        dlgItem.setTitel(rz.getLocale().getString("waehrung.key_not_found"));
                        dlgItem.setMessage(rz.getLocale().getString("waehrung.fehler_not_found"));
                        dlgItem.setIcon("warnung.png");
                        dlgItem.setOk(rz.getLocale().getString("string_ok"));
                        rz.getDialogFactory().getDialog(dlgItem);
                  }
                } else {
                    dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
                    dlgItem.setCaption(rz.getLocale().getString("waehrung.cannot_delete_titel"));
                    dlgItem.setTitel(rz.getLocale().getString("waehrung.cannot_delete_titel"));
                    dlgItem.setMessage(rz.getLocale().getString("waehrung.fehler_cannot_delete").replaceAll("%s",titel));
                    dlgItem.setIcon("warnung.png");
                    dlgItem.setOk(rz.getLocale().getString("string_ok"));
                    rz.getDialogFactory().getDialog(dlgItem);
                }
            }
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBCurrency.dbDeleteCurrency:", e.getLocalizedMessage());
        }
    }

    /**
     * dbSaveCurrency(CurrencyItem) speichert eine neue Währung oder
     * ändert eine Währung in der Datenbank ab
     */
    public int dbSaveCurrency(CurrencyItem ci){
        int ret = 0;
        PreparedStatement stm;
        try {
            if (ci.getID() == 0){
                stm = saveCurrency;
                ci.setUser(rz.getDatabase().dbGetUser());
            }
            else
            {
                stm = updateCurrency;
                stm.setInt(6,ci.getID());
                if(!db.idExists(DBFactory.TABLE_CURRENCIES,ci.getID(),true)) {
                	stm=saveCurrency;
                	ci.setID(0);
                }
            }
            stm.setString(1,  ci.getName());
            stm.setString(2, ci.getShortName());
            stm.setDouble(3, ci.getRate());
            stm.setString(4, db.dbGetDateFormatString(db.getServerTimestamp()));
            stm.setString(5, ci.getUser1());
            stm.executeUpdate();
            if (ci.getID() == 0){
                ret = db.dbGetLastInsertID("currencies");
            } else {
				ret = ci.getID();
			}
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBCurrency.dbSaveCurrency:", e.getLocalizedMessage());
            ret = 0;
        }
        return ret;
    }

    /** CurrencyListe dbGetCurrencyListeFromDB() gibt die Liste aller Währungen
     * zurück,
     */
    public CurrencyListe dbGetCurrencyListe(){
        final CurrencyListe cl = new CurrencyListe();
        ResultSet rs;
        try {
            getCurrencyListe.executeQuery();
            rs = getCurrencyListe.getResultSet();
            while (rs.next()){
                cl.addItem(new CurrencyItem(
                    rs.getInt("id"),
                    rs.getString("name"),
                    rs.getString("shortname"),
                    rs.getDouble("rate"),
                    db.getDateFromString(rs.getString("datum")),
                    rs.getString("user_1")));
            }
            if(rs != null) {
				rs.close();
			}
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBCurrency.dbGetCurrencyListe:", e.getLocalizedMessage());
        }
        return cl;
    }
    /**
     * CurrencyItem dbGetCurrency(int aKey) gibt ein CurrencyItem zur�ck
     */

    public CurrencyItem dbGetCurrency(int aKey,boolean quiet){
        CurrencyItem ci = null;
        ResultSet rs;
        try {
            if(aKey == 0) {
               ci = new CurrencyItem();
            } else {
            	if(db.idExists(DBFactory.TABLE_CURRENCIES,aKey,quiet)) {
		            getCurrency.setInt(1, aKey);
		            getCurrency.executeQuery();
		            rs = getCurrency.getResultSet();
		            if (rs.next()){
		                ci = new CurrencyItem(
		                    rs.getInt("id"),
		                    rs.getString("name"),
		                    rs.getString("shortname"),
		                    rs.getDouble("rate"),
		                    db.getDateFromString(rs.getString("datum")),
		                    rs.getString("user_1"));
		            }
            	}
            }
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBCurrency.dbGetCurrency:", e.getLocalizedMessage());
        }
        return ci;
    }

    /** void dbGetCurrencyListeCombo(JComboBox) füllt die angegebene JComboBox mit einer
     * Währungsliste und markiert eine Währung
     * @param cmb  die Combobox
     * @param aDefault   Die id der zu selektierenden währung oder -1 für Default
     */
    public void dbGetCurrencyListeCombo(JComboBox cmb,int aDefault){
        int sel;
        final CurrencyListe cl = dbGetCurrencyListe();
        cmb.removeAllItems();
        for(int i = 0 ; i < cl.size();i++){
            cmb.addItem(cl.get(i));
        }
        if(aDefault == -1) {
            sel =  rz.getOptionFactory().getOption("kalkulation.currency",1);
        } else {
            sel = aDefault;
        }
        for(int i = 0 ;i < cmb.getModel().getSize();i++){
            if(((CurrencyItem)cmb.getModel().getElementAt(i)).getID() == sel){
                cmb.setSelectedIndex(i);
                break;
            }
        }
    }
    /**
     * rechnet einen Betrag in eine Zielwährung um
     * @param value
     * @param source   CurrencyItem Quelle
     * @param dest	  CurrencyItem Ziel
     * @return betrag in der dest-Währung
     */
    public BigDecimal convert(BigDecimal value,CurrencyItem source,CurrencyItem dest) {
    	if(source.getID() == dest.getID()) {
    		return value;
    	}
		// in euro umrechnen
		if(dest.getID() == euro.getID()) {
			return value.multiply( new BigDecimal(1/source.getRate())).setScale(RzPro.CURRENCY_DEFAULT_SCALE,BigDecimal.ROUND_HALF_UP);
		}
		return value.multiply( new BigDecimal(1/source.getRate()))	.multiply(new BigDecimal(dest.getRate())).setScale(RzPro.CURRENCY_DEFAULT_SCALE,BigDecimal.ROUND_HALF_UP);
    }
}
