/*
 *  Copyright (C) database.stichwort_loeschen4 Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


package de.lunqual.rzpro.database;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Set;

import javax.swing.JComboBox;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.fenster.ZutatenDialog;
import de.lunqual.rzpro.fenster.dialoge.DialogControl;
import de.lunqual.rzpro.items.buchung.BuchungsItem;
import de.lunqual.rzpro.items.buchung.SaldenItem;
import de.lunqual.rzpro.items.deklarationen.DeklarationsListe;
import de.lunqual.rzpro.items.dialog.DialogItem;
import de.lunqual.rzpro.items.ergebnis.DependencyItem;
import de.lunqual.rzpro.items.ergebnis.DependencyListe;
import de.lunqual.rzpro.items.konform.konformItem;
import de.lunqual.rzpro.items.lager.LagerItem;
import de.lunqual.rzpro.items.rechnen.RechenItem;
import de.lunqual.rzpro.items.rezeptur.AusstattungsItem;
import de.lunqual.rzpro.items.rezeptur.BestandswarnungsItem;
import de.lunqual.rzpro.items.rezeptur.BestandswarnungsListe;
import de.lunqual.rzpro.items.rezeptur.ChangesListe;
import de.lunqual.rzpro.items.rezeptur.EanItem;
import de.lunqual.rzpro.items.rezeptur.HinweisItem;
import de.lunqual.rzpro.items.rezeptur.RezepturBuchungsItem;
import de.lunqual.rzpro.items.rezeptur.RezepturHeaderItem;
import de.lunqual.rzpro.items.rezeptur.RezepturItem;
import de.lunqual.rzpro.items.rezeptur.SteuerDatenItem;
import de.lunqual.rzpro.items.rezeptur.SteuerLagerItem;
import de.lunqual.rzpro.items.rezeptur.ZutatenListe;
import de.lunqual.rzpro.items.rezeptur.ZutatenSuchItem;
import de.lunqual.rzpro.items.stichworte.StichwortListe;
import de.lunqual.rzpro.items.versionen.VersionsItem;
import de.lunqual.rzpro.log.LogFactory;
import de.lunqual.rzpro.options.OptionFactory;

/**
 *
 * @author  lunqual
 */
public class DBRezeptur {

    RzPro           rz;
    Connection      con;  
    DBFactory       db;

    Statement           getReadOnly;

    PreparedStatement   	getRezeptur;
    PreparedStatement	  	getRezepturBuchung;
    PreparedStatement   	saveRezeptur;
    PreparedStatement   	updateRezeptur;
    PreparedStatement   	saveVerweis;
    PreparedStatement   	updateVerweis;
    PreparedStatement   	getRezepturName;
    PreparedStatement   	getRezepturID;
    PreparedStatement   	checkRezepturRezeptliste;
    PreparedStatement   	checkVerweis;
    PreparedStatement   	deleteRezeptur;
    PreparedStatement   	getVerweise;
    PreparedStatement		getBestandseinheit;
    PreparedStatement   	updateLitergewicht;
    PreparedStatement	   	updatePreis;
    PreparedStatement	   	updateExtrakt;
    PreparedStatement	   	updateStaerke;
    PreparedStatement	   	updateKalkulation;
    PreparedStatement	   	updateText;
    PreparedStatement	   	updateBestand;
    PreparedStatement	   	getBestandsSaldo;
    PreparedStatement	   	getBestandsSaldoDatum;
    PreparedStatement		updateExtraktfrei;
    PreparedStatement   	updateHinweis;
    PreparedStatement   	updateBrennwert;
    PreparedStatement	   	updateAcid;
    PreparedStatement	   	updateDeklarationen;
    PreparedStatement    	getDeklarationen;
    PreparedStatement	  	getComment;
    PreparedStatement	  	getTyp;
    PreparedStatement   	getStaerke;
    PreparedStatement   	getLosnummer;
    PreparedStatement   	getArtikelnummer;
    PreparedStatement		getStichworte;
    //Zutaten suchen
    PreparedStatement		getZutatenListe;
    PreparedStatement		getZutatenListeFertigwaren;
    PreparedStatement		getZutatenListeFertigwarenAblage;
    PreparedStatement		getZutatenListeAusstattung;
    PreparedStatement		getZutatenListeAblage;
    
    PreparedStatement		getBestandswarnung;
    PreparedStatement		getBestandswarnungAuswahl;
    PreparedStatement		getBestandswarnungEinzeln;
    PreparedStatement		checkBestellungRezeptur;
    PreparedStatement    	getAdressen;
    PreparedStatement		getMusterNummer;
    PreparedStatement		checkRezepturNameExists;
    PreparedStatement		checkRezepturNameExistsRegex;
    PreparedStatement		updateMuster;
    
    PreparedStatement		getVersionen;
    PreparedStatement		saveVersion;
    PreparedStatement		deleteVersion;
    PreparedStatement		getVersion;

    PreparedStatement		deleteVersionen;
    PreparedStatement		setRProbe;
    
    PreparedStatement		getHeader;
    PreparedStatement		updateGeaendert;
    PreparedStatement		getHinweis;
    
    PreparedStatement		getKonform;
    
    PreparedStatement		getKonformListeAlpha;
    PreparedStatement		getKonformListeAlphaReverse;
    PreparedStatement		getKonformListeDatum;
    PreparedStatement		getKonformListeDatumReverse;
    
    PreparedStatement		getRezepturFromTank;
    
    PreparedStatement		getEan;
    
    PreparedStatement		getSteuerlagerListe;
    PreparedStatement		saveSteuerLager;
    PreparedStatement		updateSteuerLager;
    PreparedStatement		checkSteuerLager;
    PreparedStatement		checkSteuerLagerBuchung;
    PreparedStatement		checkSteuerLagerInventur;
    PreparedStatement		deleteSteuerLager;
    PreparedStatement		getSteuerLager;
    PreparedStatement		getRezepturSteuerDaten;
    PreparedStatement		updateSteuerDaten;
    PreparedStatement		getRezepturFromDeklaration;
    
    PreparedStatement		updateEAN;
    
    PreparedStatement		getLager;
    
    PreparedStatement		checkNameID;
    PreparedStatement		getNameStichworte;
    
    PreparedStatement		getProbenAnzahl;
    
    PreparedStatement		getRawText;
    
    DialogItem					dlgItem;
    

    public static final int REZEPTURWASSER = -1;

    private static final int RECURSION_COUNT = 32;

    /** Creates a new instance of DBRezeptur */
    public DBRezeptur(RzPro r, DBFactory aDb, Connection aCon) {
        rz = r;
        db = aDb;
        con = aCon;
        setStatements();
        dlgItem = new DialogItem(0,"",0.0,"","","","","","",null);
    }

    private void setStatements(){
        try{
            getReadOnly     = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            getRezeptur       	= con.prepareStatement("SELECT  " +
            	"id,typ,name,verweis,comment,adressen,stichworte,deklarationen,deklarationen_alle,staerke,extraktfrei,extrakt," +
                "litergewicht,basiseinheit,basismenge," +
                "bestand,meldebestand,vorschlagsbestand,bestandseinheit,"+
                "artikelnummer,losnummer,preis,preismenge,preiseinheit,waehrung,lager,hinweis,hinweis_user,hinweis_datum,schwundsatz,"+
                "erstellt,geaendert,user_1,user_2,brennwert,brennwert_af,acid,tank,spezifikation,r_probe,tempvon,tempbis,konformDatum,konformComment,konform,spezifikation_text,bewertung,ean,steuer_artikel,sorte,steuerlager,auto_anfrage,proben_anzahl  "+
           		" FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            
            getRezepturBuchung       	= con.prepareStatement("SELECT  " +
                	"id,typ,name,verweis,staerke,extraktfrei,extrakt," +
                    "litergewicht,"+
                    "bestand,meldebestand,vorschlagsbestand,basiseinheit,bestandseinheit,schwundsatz,"+
                    "artikelnummer,losnummer,lager "+
               		" FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            getComment       	= con.prepareStatement("SELECT comment FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getBestandseinheit   	= con.prepareStatement("SELECT bestandseinheit FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            
            getTyp       	= con.prepareStatement("SELECT typ FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getDeklarationen   = con.prepareStatement("SELECT deklarationen  FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getRezepturName   = con.prepareStatement("SELECT id,name FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getRezepturID     	= con.prepareStatement("SELECT id,name FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE name=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            checkRezepturRezeptliste     	= con.prepareStatement("SELECT id,comment FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE comment REGEXP ?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            checkVerweis      	= con.prepareStatement("SELECT id,verweis FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE verweis=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getVerweise       	= con.prepareStatement("SELECT name FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE verweis=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            deleteRezeptur    	= con.prepareStatement("DELETE FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            saveVerweis         	= con.prepareStatement("INSERT INTO " + DBFactory.TABLE_REZEPTLISTE + " (typ,name,verweis,adressen,stichworte,user_1,user_2,erstellt,geaendert) VALUES (?,?,?,?,?,?,?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateVerweis     	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set typ=?,name=?,verweis=?,adressen=?,stichworte=?,user_1=?,user_2=?,erstellt=?,geaendert=? where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateLitergewicht 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set litergewicht=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updatePreis		 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set preis=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateExtrakt		 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set extrakt=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateExtraktfrei	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set extraktfrei=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateAcid			 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set acid=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateDeklarationen = con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set deklarationen_alle=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateStaerke	 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set staerke=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateKalkulation 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set preis=?,preiseinheit=?,preismenge=?,waehrung=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateText		 		= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set comment=?,staerke=?,user_2=?,geaendert=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateBestand	 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set bestand=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateGeaendert		 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set geaendert=?,user_2=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            
            getBestandsSaldo  		= con.prepareStatement("SELECT rezeptliste.bestandseinheit,sum(rest_liter) as rest_liter,sum(rest_kg) as rest_kg,sum(rest_la) as rest_la " +
            		 " FROM " +  DBFactory.TABLE_BUCHUNG + " LEFT JOIN " + DBFactory.TABLE_REZEPTLISTE + " on rezeptliste.id = buchungen.rezeptur_id " + 
            		 "  WHERE rezeptur_id=? and rest <>0 GROUP BY  rezeptur_id",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            getBestandsSaldoDatum  		= con.prepareStatement("SELECT buchungen.datum,rezeptliste.bestandseinheit,sum(rest_liter) as rest_liter,sum(rest_kg) as rest_kg,sum(rest_la) as rest_la " +
            		 " FROM " +  DBFactory.TABLE_BUCHUNG + " LEFT JOIN " + DBFactory.TABLE_REZEPTLISTE + " on rezeptliste.id = buchungen.rezeptur_id " + 
            		 "  WHERE rezeptur_id=? and buchungen.datum <=? and rest <>0 GROUP BY  rezeptur_id",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

            updateHinweis	 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set hinweis=?,hinweis_user=?,hinweis_datum=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateBrennwert 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set brennwert=?,brennwert_af=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            getBestandswarnung          = con.prepareStatement("SELECT  rezeptliste.id,rezeptliste.stichworte,staerke,name,bestandseinheit,vorschlagsbestand-bestand as diff ,sum(bestellung.rest) as rest, bestand FROM " + DBFactory.TABLE_REZEPTLISTE + " left join bestellung on rezeptliste.id=bestellung.rezeptur  WHERE meldebestand <>0 and bestand <= meldebestand   group by rezeptliste.id  order by diff desc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getBestandswarnungAuswahl          = con.prepareStatement("SELECT  rezeptliste.id,rezeptliste.stichworte,staerke,name,bestandseinheit,vorschlagsbestand-bestand as diff ,sum(bestellung.rest) as rest, bestand FROM " + DBFactory.TABLE_REZEPTLISTE + " left join bestellung on rezeptliste.id=bestellung.rezeptur  WHERE meldebestand <>0 and bestand <= meldebestand  and stichworte not regexp ?   group by rezeptliste.id  order by diff desc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            getBestandswarnungEinzeln          = con.prepareStatement("SELECT  rezeptliste.id,rezeptliste.stichworte,rezeptliste.auto_anfrage,staerke,name,bestandseinheit,vorschlagsbestand-bestand as diff ,sum(bestellung.rest) as rest, bestand FROM " + DBFactory.TABLE_REZEPTLISTE + " left join bestellung on rezeptliste.id=bestellung.rezeptur  WHERE meldebestand <>0 and bestand <= meldebestand  and rezeptliste.auto_anfrage <> 0 and rezeptliste.id=?   group by rezeptliste.id  order by diff desc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
            
            saveRezeptur      = con.prepareStatement(
                "INSERT INTO " + DBFactory.TABLE_REZEPTLISTE + " (" +
                "typ,name,verweis,comment,adressen,stichworte,deklarationen,deklarationen_alle,staerke,extraktfrei,extrakt," +
                "litergewicht,basiseinheit,basismenge," +
                "bestand,meldebestand,vorschlagsbestand,bestandseinheit,"+
                "artikelnummer,losnummer,preis,preismenge,preiseinheit,waehrung,lager,hinweis,schwundsatz,"+
                "erstellt,geaendert,user_1,user_2,brennwert,brennwert_af,acid,tank,spezifikation,r_probe,tempvon,tempbis,konformDatum,konformComment,konform,"+
                "spezifikation_text,bewertung,ean,steuer_artikel,sorte,steuerlager,auto_anfrage, proben_anzahl " +
                ")" +
                " VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)",
                ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
            updateRezeptur      = con.prepareStatement(
                "UPDATE " + DBFactory.TABLE_REZEPTLISTE + " " +
                "set typ=?,name=?,verweis=?,comment=?,adressen=?,stichworte=?,deklarationen=?,deklarationen_alle=?,staerke=?,extraktfrei=?,extrakt=?," +
                "litergewicht=?,basiseinheit=?,basismenge=?," +
                "bestand=?,meldebestand=?,vorschlagsbestand=?,bestandseinheit=?,"+
                "artikelnummer=?,losnummer=?,preis=?,preismenge=?,preiseinheit=?,waehrung=?,lager=?,hinweis=?,schwundsatz=?,"+
                "erstellt=?,geaendert=?,user_1=?,user_2=?,brennwert=?,brennwert_af=?,acid=?,tank=?,spezifikation=?,r_probe=?,tempvon=?,tempbis=?,konformDatum=?," +
                "konformComment=?,konform=?,spezifikation_text=?,bewertung=?,ean=?,steuer_artikel=?,sorte=?,steuerlager=?,auto_anfrage=?,proben_anzahl=? "+
                " WHERE id=?",
                ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);

           getZutatenListe  						= con.prepareStatement("SELECT id,name,artikelnummer,staerke,extrakt,stichworte,litergewicht,bestand,meldebestand FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE (typ=1 OR typ=3) AND  name REGEXP ? ORDER BY name asc LIMIT ?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getZutatenListeFertigwaren		= con.prepareStatement("SELECT id,name,artikelnummer,staerke,extrakt,stichworte,litergewicht,bestand,meldebestand FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE (typ=1 OR typ=3) AND  name REGEXP ? AND stichworte regexp ?  ORDER BY name asc LIMIT ?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getZutatenListeFertigwarenAblage		= con.prepareStatement("SELECT id,name,artikelnummer,staerke,extrakt,stichworte,litergewicht,bestand,meldebestand FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE (typ=1 OR typ=3) AND  name REGEXP ? AND stichworte regexp ?  and stichworte not regexp ? ORDER BY name asc LIMIT ?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

           
           getZutatenListeAusstattung  	= con.prepareStatement("SELECT id,name,artikelnummer,staerke,extrakt,stichworte,litergewicht,bestand,meldebestand FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE typ = 4 AND  name REGEXP ? ORDER BY name asc LIMIT ?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

           getZutatenListeAblage							= con.prepareStatement("SELECT id,name,artikelnummer,staerke,extrakt,stichworte,litergewicht,bestand,meldebestand FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE (typ=1 OR typ=3) AND  name REGEXP ? and stichworte not regexp ? ORDER BY name asc LIMIT ?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
  
           getStaerke       	= con.prepareStatement("SELECT staerke FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getLosnummer 	= con.prepareStatement("SELECT losnummer,artikelnummer FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getArtikelnummer 	= con.prepareStatement("SELECT artikelnummer FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getAdressen       	= con.prepareStatement("SELECT adressen FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getMusterNummer       	= con.prepareStatement("SELECT muster FROM " + DBFactory.TABLE_MUSTERNUMMER + " LIMIT 1 ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           checkRezepturNameExists       	= con.prepareStatement("SELECT id FROM " + DBFactory.TABLE_REZEPTLISTE + " where name=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           checkRezepturNameExistsRegex       	= con.prepareStatement("SELECT id FROM " + DBFactory.TABLE_REZEPTLISTE + " where name regexp ? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

           updateMuster	 	= con.prepareStatement("update " + DBFactory.TABLE_MUSTERNUMMER + " set muster=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
           getVersionen      	= con.prepareStatement("SELECT id,rezeptur,user,erstellt,version,bezeichnung FROM " + DBFactory.TABLE_VERSIONEN + " WHERE rezeptur=? ORDER BY erstellt ASC",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           saveVersion 			= con.prepareStatement("INSERT INTO " + DBFactory.TABLE_VERSIONEN + " (rezeptur,user,erstellt,version,bezeichnung) VALUES (?,?,?,?,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
           deleteVersion 		= con.prepareStatement("DELETE FROM " + DBFactory.TABLE_VERSIONEN + " where id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
           getVersion      	= con.prepareStatement("SELECT id,rezeptur,user,erstellt,version,bezeichnung FROM " + DBFactory.TABLE_VERSIONEN + " WHERE id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           deleteVersionen 		= con.prepareStatement("DELETE FROM " + DBFactory.TABLE_VERSIONEN + " where rezeptur=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
           getStichworte 	= con.prepareStatement("SELECT stichworte FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           setRProbe	 	= con.prepareStatement("update " + DBFactory.TABLE_REZEPTLISTE + " set r_probe=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
           getHeader 	= con.prepareStatement("SELECT id,name,erstellt,geaendert,user_1,user_2,staerke,artikelnummer FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
                        
           getHinweis 	= con.prepareStatement("SELECT id,hinweis,hinweis_user,hinweis_datum FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getKonform       	= con.prepareStatement("SELECT id,name,konform,konformDatum,konformComment FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getKonformListeAlpha	= con.prepareStatement("SELECT id,name,konform,konformDatum,konformComment FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE konform <> 0 order by name asc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getKonformListeAlphaReverse	= con.prepareStatement("SELECT id,name,konform,konformDatum,konformComment FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE konform <> 0 order by name desc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getKonformListeDatum	= con.prepareStatement("SELECT id,name,konform,konformDatum,konformComment FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE konform <> 0 order by konformDatum asc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getKonformListeDatumReverse	= con.prepareStatement("SELECT id,name,konform,konformDatum,konformComment FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE konform <> 0 order by konformDatum desc",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getRezepturFromTank	= con.prepareStatement("SELECT id FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE tank regexp ? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getEan       	 	= con.prepareStatement("SELECT ean FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           updateEAN       	 	= con.prepareStatement("UPDATE " + DBFactory.TABLE_REZEPTLISTE + " set ean=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
           getSteuerlagerListe 	= con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_STEUERLAGER + " ORDER BY id asc ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           saveSteuerLager	 	= con.prepareStatement("INSERT INTO " + DBFactory.TABLE_STEUERLAGER + " (`system`,bezeichnung) VALUES (0,?)",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
           updateSteuerLager 	= con.prepareStatement("UPDATE " + DBFactory.TABLE_STEUERLAGER + " set `system`=0,bezeichnung=? WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
           checkSteuerLager	 	= con.prepareStatement("SELECT id FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE steuerlager=? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           checkSteuerLagerBuchung	 	= con.prepareStatement("SELECT id FROM " + DBFactory.TABLE_BUCHUNG + " WHERE steuerlager=? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           checkSteuerLagerInventur	 	= con.prepareStatement("SELECT id FROM " + DBFactory.TABLE_INVENTURZEILE + " WHERE steuerlager=? LIMIT 1",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           deleteSteuerLager 	= con.prepareStatement("DELETE FROM " + DBFactory.TABLE_STEUERLAGER + " WHERE ID =?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE);
           getSteuerLager	 	= con.prepareStatement("SELECT * FROM " + DBFactory.TABLE_STEUERLAGER + " WHERE id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getRezepturSteuerDaten 	= con.prepareStatement("SELECT rezeptliste.steuer_artikel,rezeptliste.sorte,rezeptliste.steuerlager,sorten.bezeichnung,steuer_lager.bezeichnung,rezeptliste.lager,lager.id,lager.name FROM " + DBFactory.TABLE_REZEPTLISTE + " left join lager on rezeptliste.lager=lager.id left join sorten on rezeptliste.sorte=sorten.id left join steuer_lager on rezeptliste.steuerlager=steuer_lager.id WHERE rezeptliste.id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           updateSteuerDaten 	= con.prepareStatement("UPDATE " + DBFactory.TABLE_REZEPTLISTE + " set steuer_artikel=?,sorte=?,steuerlager=? WHERE ID =?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE); 
           getRezepturFromDeklaration	 	= con.prepareStatement("SELECT id FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE deklarationen regexp ?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getLager	 	= con.prepareStatement("SELECT lager FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           checkNameID      	= con.prepareStatement("SELECT id,name FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE name regexp ? and id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getNameStichworte   	= con.prepareStatement("SELECT id,name,stichworte FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE name=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getProbenAnzahl  	= con.prepareStatement("SELECT proben_anzahl FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=? ",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
           getRawText 			= con.prepareStatement("SELECT comment FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id=?",ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);

           
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.setStatements", e.getLocalizedMessage());
        }
    }

    public boolean isRechenRezeptur(int id) {
    	boolean ret = false;
    	if(id != 0) {
    	try {
	    		getRawText.setInt(1, id);
	    		final ResultSet rs = getRawText.executeQuery();
	    		if(rs.next()) {
		    		String raw = rs.getString("comment");
		    		String[] lines = raw.split("\\n");
					for(String line:lines) {
						if(line.contains("[[")) {
							int i = Integer.valueOf(line.substring(line.indexOf("[[")+2,line.indexOf("]]")));
							if (i > 0) {
								return true;
							}
						}
					}
				}
			}   catch (final Exception e){
				rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,isRechenRezeptur" , e.getLocalizedMessage());
			}
	   	}
    	return ret;
    }
    
    public int getProbenAnzahl(int id) {
    	int ret = 0;
    	try {
    		getProbenAnzahl.setInt(1, id);
    		final ResultSet rs = getProbenAnzahl.executeQuery();
    		if(rs.next()) {
    			ret = rs.getInt("proben_anzahl");
    		}
		}  catch (final Exception e){
			rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getProbenanzahl" , e.getLocalizedMessage());
		}
    	
    	return ret;
    }
    
    public String getAlleRezepturen(int id) {
    	String ret = Integer.valueOf(id)+",";
    	HashMap<Integer,Integer> map = new HashMap<Integer,Integer>();
    	getRawNumbers(map, id);
    	Set keys = map.keySet();
    	for(Object key: keys){
    	    ret += map.get(key) + ",";
    	}
		if(ret.endsWith(",")) {
			ret = ret.substring(0,ret.length()-1);
		}		
    	return ret;
    }
    
    private void getRawNumbers(HashMap<Integer,Integer> map ,int id){
		RezepturItem ri = rz.getDatabase().getRezeptur().dbGetRezeptur(id, true);
		if(ri != null) {
			String raw = ri.getRaw();
			if(!raw.trim().equals("")) {
				String[] lines = raw.split("\\n");
				for(String line:lines) {
					if(line.contains("[[")) {
						int i = Integer.valueOf(line.substring(line.indexOf("[[")+2,line.indexOf("]]")));
						if(!map.containsKey(i)) {
							map.put(i, i);
							getRawNumbers(map, i);
						}
					}
				}
			}
		}
    }
    
    public String getLagerName(int rezeptur) {
    	String ret = "";
    	try {
    		if(rezeptur > 0) {
    			getLager.setInt(1, rezeptur);
    			final ResultSet rs = getLager.executeQuery();
    			if(rs.next()) {
    				int lager = rs.getInt("lager");
    				if(lager != 0) {
    					ret = ((LagerItem)rz.getDatabase().getLager().dbGetLager(lager,true)).getName();
    				}	
    			}
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getLagerName", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    public void updateEAN(int rezeptur,String prefix,String ean) {
    	try {
    		String strEan = "";
    		if(rezeptur != 0 && !prefix.equals("") && !ean.equals("")){
	    		getEan.setInt(1, rezeptur);
	    		final ResultSet rs = getEan.executeQuery();
	    		if(rs.first()) {
	    			strEan = rs.getString("ean");
	    			strEan += (strEan.trim().equals("")?"":"\n") + prefix + ":" + ean;
	    			updateEAN.setString(1, strEan);
	    			updateEAN.setInt(2, rezeptur);
	    			updateEAN.executeUpdate();
	    		
	    		}
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.updateEAN", e.getLocalizedMessage());
        }
    }
    
    
    public ArrayList<Integer> getRezepturenFromDeklaration(int deklaration){
    	ArrayList<Integer> ret = new ArrayList<Integer>();
    	HashMap<Integer,Integer> liste = new HashMap<Integer,Integer>();
    	try {
    		if(deklaration >0) {
    			String regex = "<" + String.valueOf(deklaration) + ">";
    			getRezepturFromDeklaration.setString(1, regex);
    			final ResultSet rs = getRezepturFromDeklaration.executeQuery();
    			while(rs.next()) {
    				int id = rs.getInt("id");
    				if(!liste.containsKey(id)) {
    					liste.put(id,id);
    				}
    			}
    			if(!liste.isEmpty()) {
    				ArrayList<Integer> level0 = new ArrayList<Integer>();
    				for (int key : liste.keySet()) {
    				    level0.add(key);
    				}
    				for(int d:level0) {
    					final DependencyListe el = rz.getDatabase().getErgebnis().dbGetErgebnisListeByID(d,true);
    					HashMap<Integer,Integer> level1 = new HashMap<Integer,Integer>();
    					if(el != null && el.size()>0) {
    						for(int i=0;i<el.size();i++) {
    							DependencyItem di = el.getItem(i);
    							if(di != null) {
    								if(!level1.containsKey(di.getId())) {
    									level1.put(di.getId(), di.getId());
    								}
    							}
    						}
    						for(int key:level1.keySet()) {
    							if(!liste.containsKey(key)) {
    								liste.put(key, key);
    							}
    						}
    						
    					}
    				}
    			}
    		}
    		for(int key:liste.keySet()) {
    			ret.add(key);
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getRezepturenFromDeklaration", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    public void updateSteuerDaten(int rezeptur,String steuerArtikel,int sorte,int steuerlager) {
    	try {
    		updateSteuerDaten.setString(1, steuerArtikel);
    		updateSteuerDaten.setInt(2, sorte);
    		updateSteuerDaten.setInt(3, steuerlager);
    		updateSteuerDaten.setInt(4, rezeptur);
    		updateSteuerDaten.executeUpdate();
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.updateSteuerDatenFelder", e.getLocalizedMessage());
        }
    }
    
    public void updateSteuerDaten(BuchungsItem bi) {
    	try {
    		updateSteuerDaten.setString(1, bi.getSteuer_artikel());
    		updateSteuerDaten.setInt(2, bi.getSorte());
    		updateSteuerDaten.setInt(3, bi.getSteuerlager());
    		updateSteuerDaten.setInt(4, bi.getRezeptur_id());
    		updateSteuerDaten.executeUpdate();
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.updateSteuerDaten", e.getLocalizedMessage());
        }
    }
    
    public void setSteuerlagerCmb(JComboBox cmb,int value){
        cmb.removeAllItems();
        ArrayList<SteuerLagerItem> ll = getSteuerlagerListe();
        for(int i = 0 ; i < ll.size();i++){
            cmb.addItem(ll.get(i));
        }
        setSteuerlagerIndex(cmb,value);
    }
    
    public void setSteuerlagerIndex(JComboBox cmb,int value) {
    	for(int i = 0 ;i < cmb.getModel().getSize();i++){
            if(((SteuerLagerItem)cmb.getModel().getElementAt(i)).getId() == value){
                cmb.setSelectedIndex(i);
                break;
            }
        }
    }
    
    public SteuerDatenItem getSteuerDaten(int rezeptur_id) {
    	SteuerDatenItem ret = new SteuerDatenItem();
    	try {
    		getRezepturSteuerDaten.setInt(1, rezeptur_id);
    		final ResultSet rs = getRezepturSteuerDaten.executeQuery();
    		if(rs.next()) {
    			ret = new SteuerDatenItem(
    					rs.getString("rezeptliste.steuer_artikel"),
    					rs.getInt("rezeptliste.sorte"),
    					rs.getInt("rezeptliste.steuerlager"),
    					rs.getString("sorten.bezeichnung"),
    					rs.getString("steuer_lager.bezeichnung"),
    					rs.getInt("rezeptliste.lager"),
    					rs.getString("lager.name")
    					);
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getSteuerDaten", e.getLocalizedMessage());
        }
    	return ret;
    }
    public SteuerLagerItem getSteuerLager(int id) {
    	SteuerLagerItem ret = new SteuerLagerItem();
    	try {
    		if(id != 0) {
    			getSteuerLager.setInt(1, id);
    			final ResultSet rs = getSteuerLager.executeQuery();
    			if(rs.next()) {
    				ret = new SteuerLagerItem(
        					rs.getInt("id"),
        					rs.getInt("system") != 0?true:false,
        					rs.getString("bezeichnung")
        					);
    			}
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.checkSteuerlager", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    /*
     * prüft ob ein Steuerlager in Rezepturen,Buchungen oder einer Inventur verwendet wird
     */
    public boolean checkSteuerLager(SteuerLagerItem si) {
    	boolean rr = false,rb = false,ri = false;
    	try {
    		if(si != null) {
    			checkSteuerLager.setInt(1, si.getId());
    			final ResultSet rs = checkSteuerLager.executeQuery();
    			if(rs.next()) {
    				rr = true;
    			}
    			checkSteuerLagerBuchung.setInt(1, si.getId());
    			final ResultSet rs1 = checkSteuerLagerBuchung.executeQuery();
    			if(rs1.next()) {
    				rb = true;
    			}
    			checkSteuerLagerInventur.setInt(1, si.getId());
    			final ResultSet rs2 = checkSteuerLagerInventur.executeQuery();
    			if(rs2.next()) {
    				ri = true;
    			}
    		}
    	}catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.checkSteuerlager", e.getLocalizedMessage());
        }
    	if(rr == true && rb == true && ri == true) {
    		return true;
    	}else {
    		return false;
    	}
     }
    
    public boolean deleteSteuerLager(SteuerLagerItem si) {
    	boolean ret = true;
    	try {
    		if(si != null && !si.isSystem()) {
    			deleteSteuerLager.setInt(1, si.getId());
    			deleteSteuerLager.executeUpdate();
    		} else {
    			ret = false;
    		}
    	}
    	catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.deleteSteuerlager", e.getLocalizedMessage());
        }
    	return ret;
    }
    
    public int saveSteuerLager(SteuerLagerItem si) {
    	int ret = 0;
    	try {
    		if(si != null && si.getId() == 0 && !si.getBezeichnung().trim().equals("")) {
    			saveSteuerLager.setString(1, si.getBezeichnung());
    			saveSteuerLager.executeUpdate();
    			ret = db.dbGetLastInsertID(DBFactory.TABLE_STEUERLAGER);
    			si.setId(ret);
    		} else if (si != null && si.getId() != 0 && !si.getBezeichnung().trim().equals("")){
    			updateSteuerLager.setString(1, si.getBezeichnung());
    			updateSteuerLager.setInt(2, si.getId());
    			updateSteuerLager.executeUpdate();
    			ret = si.getId();
    			
    		}
    		
		}  catch (final Exception e){
			rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,saveSteuerLager " , e.getLocalizedMessage());
		}
		return ret;
    }
    public ArrayList<SteuerLagerItem> getSteuerlagerListe(){
    	ArrayList<SteuerLagerItem> ret = new ArrayList<SteuerLagerItem>(); 
    	try{
    		final ResultSet rs = getSteuerlagerListe.executeQuery();
    		while(rs.next()) {
    			ret.add(new SteuerLagerItem(
    					rs.getInt("id"),
    					rs.getInt("system") != 0?true:false,
    					rs.getString("bezeichnung")
    					));
    			
    		}
    	}catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getSteuerlagerListe", e.getLocalizedMessage());
        }
    	return ret;
    }
    
	public void getEanListeCmb(JComboBox cmb,int rezepturID){
		String ean = "";
		cmb.removeAllItems();
		cmb.addItem(new EanItem(rz.getLocale().getString("ean.liste.0"),""));
		if(rezepturID !=0) {
			try {
	    		getEan.setInt(1, rezepturID);
	    		final ResultSet rs = getEan.executeQuery();
	    		if(rs.next()) {
	    			 ean = rs.getString("ean");
	    			 if(ean != null && !ean.equals("")) {
	    					String[] l = ean.split("\\n");
	    					for(String zeile:l) {
	    						if(!zeile.trim().equals("") ) {
	    							if(!zeile.trim().startsWith("#")) {
	    								if(zeile.contains(":")) {
	    									try {
	    										cmb.addItem(new EanItem(zeile.split(":")[0].trim(),zeile.split(":")[1].trim()));
	    									}catch (Exception e) {
	    										
	    									}
	    								} else {
	    									cmb.addItem(new EanItem("",zeile.trim()));
	    								}
	    							}
	    						}
	    					}
	    				}
	    		}
			}  catch (final Exception e){
				rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getRezepturFromTank" , e.getLocalizedMessage());
			}
		}
	}
    
    public int getRezepturFromTank(String tank) {
    	int ret = 0;
    	try {
    		getRezepturFromTank.setString(1, tank);
    		final ResultSet rs = getRezepturFromTank.executeQuery();
    		if(rs.next()) {
    			ret = rs.getInt("id");
    		}
		}  catch (final Exception e){
			rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getRezepturFromTank" , e.getLocalizedMessage());
		}
    	
    	return ret;
    }
    
    public ArrayList <konformItem> getKonformListe(boolean alpha,boolean reverse){
    	ArrayList <konformItem> ret = new ArrayList<konformItem>();
    	try {
    		PreparedStatement stm=null;
    		if(alpha == true && reverse == true)stm = getKonformListeAlphaReverse;
    		if(alpha == true && reverse == false)stm = getKonformListeAlpha;
    		if(alpha == false && reverse == true)stm = getKonformListeDatumReverse;
    		if(alpha == false && reverse == false)stm = getKonformListeDatum;
    		if(stm != null) {
    			final ResultSet rs = stm.executeQuery();
    			while (rs.next()) {
    				ret.add(getKonformFromRS(rs));
    			}
    		}
		}  catch (final Exception e){
			rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getKonformListe" , e.getLocalizedMessage());
		}
    	return ret;
    }
    
    public konformItem isKonform(int id) {
    	konformItem ret = null;
    	try {
	    	if(id != 0) {
	    		getKonform.setInt(1, id);
    			final ResultSet rs = getKonform.executeQuery();
    			if(rs.next()) {
    				ret = getKonformFromRS(rs);
    			}
	    	}
		}  catch (final Exception e){
			rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getKonform" , e.getLocalizedMessage());
		}
    	return ret;
    }
    
    private konformItem getKonformFromRS(ResultSet rs) {
    	konformItem ret = new konformItem();
    	try {
				if(rs.getInt("konform") != 0) {
					ret.setKonform(true);
					ret.setRezeptur_id(rs.getInt("id"));
					ret.setRezepturBezeichnung(rs.getString("name"));
                    Date datum = (rs.getString("konformDatum") != null) ?db.getDateFromString(rs.getString("konformDatum")):null;
                    if(datum != null) {
                    	ret.setDatum(datum);
                    }
                    ret.setBemerkungen(rs.getString("konformComment"));
				} 
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getKonformFromRs " , e.getLocalizedMessage());
    	}
    	return ret;
    }

    public HinweisItem getHinweis(int id) {
    	HinweisItem ret = null;
    	try {
    		if(id != 0) {
    			getHinweis.setInt(1, id);
    			final ResultSet rs = getHinweis.executeQuery();
    			if(rs.next()) {
    				if(!rs.getString("hinweis").equals("")) {
	    				ret=new HinweisItem(
	    						id,
	    						rs.getString("hinweis"),
	    						rs.getString("hinweis_user"),
	    						db.getDateFromString(rs.getString("hinweis_datum"))
	    						);
    				}
    			}
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getHinweis " , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public RezepturHeaderItem getHeader(int id) {
    	RezepturHeaderItem ret=null;
    	try {
    		if(id != 0) {
    			getHeader.setInt(1, id);
    			final ResultSet rs = getHeader.executeQuery();
    			if(rs.next()) {
    				ret=new RezepturHeaderItem(
    						rs.getInt("id"),
    						rs.getString("name"),
    						db.getDateFromString(rs.getString("erstellt")),
    						db.getDateFromString(rs.getString("geaendert")),
    						rs.getString("user_1"),
    						rs.getString("user_2"),
    						rs.getDouble("staerke"),
    						rs.getString("artikelnummer")
    						);
    			}
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getHeader " , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    
    public void setRProbe(int id,boolean p) {
    	try {
    		if(id != 0) {
    			ChangesListe liste = new ChangesListe(rz,id);
    			setRProbe.setInt(1,p==false?0:1);
    			setRProbe.setInt(2,id);
    			setRProbe.executeUpdate();
    			rz.getDatabase().getRezeptur().updateGeaendert(id);
    			liste.commit();
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,setRProbe " , e.getLocalizedMessage());
    	}
    }
    
    
    public String getStichworte(int id) {
    	String ret="";
    	try {
    		if(id != 0) {
    			getStichworte.setInt(1,id);
    			ResultSet rs = getStichworte.executeQuery();
    			if(rs.next()) {
    				ret = rs.getString("stichworte");
    			}
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getStichworte " , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    /**
     * Versionen
     * -----------------------------
     */
    public VersionsItem getVersion(int id ) {
    	VersionsItem ret=  null;
    	try {
    		if(id >0) {
    			getVersion.setInt(1,id);
    			final ResultSet rs = getVersionen.executeQuery();
    			while (rs.next()) {
    				ret =new VersionsItem(
    						rs.getInt("id"),
    						rs.getInt("rezeptur"),
    						rs.getString("user"),
    	                    db.getDateFromString(rs.getString("erstellt")),
    						rs.getString("version"),
    						rs.getString("bezeichnung")
    				);
    			}
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getVersion (single)" , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public ArrayList<VersionsItem> getVersionen(int rezeptur,String original,String user,Date erstellt,String bezeichnung) {
    	ArrayList<VersionsItem> ret= new ArrayList<VersionsItem>();
    	try {
    			ret.add(new VersionsItem(
    					-1,
    					rezeptur,
    					rz.getLocale().getString("rezeptur.version_original") + " ("+user+")",
    					erstellt,
    					original,
    					bezeichnung
    			));
    			getVersionen.setInt(1,rezeptur);
    			final ResultSet rs = getVersionen.executeQuery();
    			while (rs.next()) {
    				ret.add(new VersionsItem(
    						rs.getInt("id"),
    						rezeptur,
    						rs.getString("user"),
    	                    db.getDateFromString(rs.getString("erstellt")),
    						rs.getString("version"),
    						rs.getString("bezeichnung")
    				));
    				
    			}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getVersionen " , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public void setVersionenCMB(int rezeptur,String original,String user,Date erstellt,String bezeichnung,JComboBox cmb) {
    	 cmb.removeAllItems();
    	 ArrayList<VersionsItem> l=getVersionen(rezeptur,original,user,erstellt,bezeichnung);
    	 for (int i =0;i<l.size();i++){
    		 cmb.addItem(l.get(i));
    	 }
    	 cmb.setSelectedIndex(0);
    }
    
    
    public VersionsItem saveVersion(int rezeptur,String version,String bezeichnung) {
    	VersionsItem ret=null;
    	try {
    		if(rezeptur != 0 && !version.trim().equals("")) {
    			saveVersion.setInt(1,rezeptur);
    			saveVersion.setString(2,db.dbGetUser());
    			saveVersion.setString(3,db.dbGetDateFormatString(db.getServerTimestamp()));
    			saveVersion.setString(4,version);
    			saveVersion.setString(5,bezeichnung);
    			saveVersion.executeUpdate();
    			ret = getVersion(db.dbGetLastInsertID(DBFactory.TABLE_VERSIONEN));
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,saveVersion " , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public void deleteVersionen(int rezeptur ) {
    	try {
    		if(rezeptur > 0) {
    			deleteVersionen.setInt(1,rezeptur);
    			deleteVersionen.executeUpdate();
    			db.getChanges().delete(rezeptur);
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,deleteVersion " , e.getLocalizedMessage());
    	}
    }
    
    public void deleteVersion(int id) {
    	try {
    		if(id != 0) {
    			deleteVersion.setInt(1,id);
    			deleteVersion.executeUpdate();
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,deleteVersion " , e.getLocalizedMessage());
    	}
    }
    
    /**
     * ---------------------------
     */
    public void setMuster(String muster) {
    	try {
    		if(!muster.trim().equals("")) {
    			updateMuster.setString(1,muster.trim());
    			updateMuster.executeUpdate();
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,setMuster " , e.getLocalizedMessage());
    	}
    }
    
    public  boolean checkRezepturNameExists(String name,boolean regex) {
    	boolean ret = false;
    	try {
    		PreparedStatement stm;
    		if(regex) {
    			stm = checkRezepturNameExistsRegex;
    		} else {
    			stm = checkRezepturNameExists;    			
    		}
    		stm.setString(1, name);
    		final ResultSet rs = stm.executeQuery();
    		if(rs.next()) {
    				ret=true;
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,checjkRezepturNameExists " , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public  boolean checkNameID(String name,int id) {
    	boolean ret = false;
    	try {
    		checkNameID.setString(1,name);
    		checkNameID.setInt(2,id);
    		final ResultSet rs = checkNameID.executeQuery();
    		if(rs.next()) {
    				ret=true;
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,checkNameID " , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    public String getNameStichworte(String name) {
    	String ret = "";
    	try {
    		getNameStichworte.setString(1, name);
    		final ResultSet rs = getNameStichworte.executeQuery();
    		while (rs.next()){
    			ret += rz.getLocale().getString("rezeptur.name_exists_dialog_message").replaceAll("%m",Integer.toString(rs.getInt("id")))
    					.replaceAll("%n", rs.getString("name"))
    					.replace("%s",db.getSTW().dbGetStichwortListe(rs.getString("stichworte")).toHTMLString());
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getNameStichworte " , e.getLocalizedMessage());
    	}
    	return ret;
    }
    
    
    private String getMusternummerVorschlag(String muster,String losnummer,String artikelnummer,int n) {
        String ret = "";
        final int nextNumber=0;
        String template=muster;
        final Date now = db.getServerTimestamp();
        final SimpleDateFormat monat = new SimpleDateFormat("MM");
        final SimpleDateFormat jahr2 = new SimpleDateFormat("yy");
        final SimpleDateFormat jahr4 = new SimpleDateFormat("yyyy");
        final SimpleDateFormat tag=new SimpleDateFormat("dd");
        final SimpleDateFormat stunde= new SimpleDateFormat("H");
        final SimpleDateFormat minute = new SimpleDateFormat("mm");
        final SimpleDateFormat julian = new SimpleDateFormat("D");
        /* templates ersetzen
         * $w = Tag des Jahres dezimal
         * $W = Tag des Jahres duodezimal
         * $m = Monat
         * $d = Tag der Herstellung
         * $y = Jahr (2-stellig)
         * $Y = Jahr (4-stellig)
         * $h = Stunde
         * $M = minute
         * $l = losnummer-vorlage aus rezeptur
         * $a = Artikelnummer
         * $u = Benutzer (vollständig)
         * $U = Benutzer (nur 1 Buchstabe)
         * $b = Anfangsbuchstaben der Rezeptur
         * $4 = die ersten 4 Buchstaben der Rezeptur
         * $X = laufende nummer
        */
        String ab ="";
        String ab4 = "";

        ret = template.replaceAll("\\$n",String.valueOf(nextNumber))
             .replaceAll("\\$w",julian.format(now))
            .replaceAll("\\$W",rz.getDatabase().getKalender().getDate20(Integer.valueOf(julian.format(now))))
        	.replaceAll("\\$m",monat.format(now))
        	.replaceAll("\\$y",jahr2.format(now))
        	.replaceAll("\\$Y",jahr4.format(now))
        	.replaceAll("\\$h",stunde.format(now))
        	.replaceAll("\\$M",minute.format(now))
        	.replaceAll("\\$d",tag.format(now))
        	.replaceAll("\\$l",losnummer.trim())
        	.replaceAll("\\$a",artikelnummer.trim())
        	.replaceAll("\\$b",ab)
        	.replaceAll("\\$4",ab4)
        	.replaceAll("\\$u",db.dbGetUser())
        	.replaceAll("\\$X", String.format("%02d",n))
        	.replaceAll("\\$U",String.valueOf(db.dbGetUser().charAt(0)));
        // jetzt noch alle übriggebliebenenen Verbotenen Zeichen entfernen
        ret = rz.getOptionFactory().deleteInvalidChars(ret);
        return ret;
    }
    
    public String getMusterNummer(String losnummer,String artikel) {
    	String ret ="";
    	String m = getMuster();
    	if(!m.equals("")) {
    		boolean weiter = true;
    		int i =0;
    		while(weiter) {
    			i += 1;
    			if (i > 50) {
    				weiter = false;
    				ret = "";
    				break;
    			}
    			String m1=getMusternummerVorschlag(m,losnummer,artikel,i);
    			if(!checkRezepturNameExists(m1,true)) {
    				ret=m1;
    				weiter=false;
    				break;
    			}
    		}
    	}
    	return ret;
    }
    
    /**
     * Musternummer holen
     */
    public String getMuster() {
    	String ret = "";
    	try {
    		final ResultSet rs = getMusterNummer.executeQuery();
    		if(rs.next()) {
    				ret=rs.getString("muster");
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getMuster " , e.getLocalizedMessage());
    	}
    	return ret.trim();
    }

    /**
     * holt die Adressliste einer Rezeptur
     */
    public StichwortListe getAdressen(int id) {
    	StichwortListe ret = null;
    	try {
    		if(id >0) {
    			getAdressen.setInt(1,id);
    			final ResultSet rs = getAdressen.executeQuery();
    			if(rs.next()) {
                    ret = db.getSTW().dbGetAdressenListe(rs.getString("adressen"));
    			}
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur,getAdressen " , e.getLocalizedMessage());
    	}
    	if (ret == null) {
    		ret = new StichwortListe();
    	}
    	return ret;
    }

    /**
     * holt eine Liste der jeweils letzten Buchung für eine Bestandswarnungsliste
     */
    public void getBestandswarnungsDatum(BestandswarnungsListe bl) {
    	try {
    		if(bl.size() > 0) {
    			final Statement stm = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
    			String items=" ( ";
    			for(int i=0;i<bl.size();i++) {
    				items += String.valueOf(bl.getItem(i).getRezeptur()) + ",";
    			}
    			items = items.substring(0,items.length()-1) + " ) ";
    			final ResultSet rs = stm.executeQuery("select rezeptur_id,datediff(max(datum),now()) as dtm from buchungen where rezeptur_id in " + items + " group by rezeptur_id");
    			while (rs.next()) {
    				bl.setDatum(rs.getInt("rezeptur_id"),rs.getInt("dtm"));
    			}
    		}
    	}  catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getBestandswarnungsDatum " , e.getLocalizedMessage());
    	}
    }
    /**
     * holt ein eingeschränktes RezepturItem für Buchungen
     */
    public RezepturBuchungsItem getRezepturBuchung(int id){
    	RezepturBuchungsItem ret = null;
    	try {
	    	if(id != 0) {
	    		getRezepturBuchung.setInt(1, id);
	    		final ResultSet rs=getRezepturBuchung.executeQuery();
	    		if(rs.next()) {
	    			ret = new RezepturBuchungsItem(
	    				rz,
	    				id,
	                    rs.getInt("typ"),
	                    rs.getString("name"),
	                    rs.getInt("verweis"),
	                    rs.getDouble("staerke"),
	                    rs.getBoolean("extraktfrei"),
	                    rs.getDouble("extrakt"),
	                    rs.getDouble("litergewicht"),
	                    getBestand(id),
	                    rs.getDouble("meldebestand"),
	                    rs.getDouble("vorschlagsbestand"),
	                    rs.getInt("basiseinheit"),
	                    rs.getInt("bestandseinheit"),
	                    rs.getString("artikelnummer"),
	                    rs.getString("losnummer"),
	                    rs.getInt("lager"),
	                    rs.getDouble("schwundsatz")
	    			);
	    		}
	    	}
    	}
    	catch (final Exception e){
    		rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBABuchung.getRezepturBuchung " , e.getLocalizedMessage());
    		ret = null;
    	}
    	return ret;
    }

    public BestandswarnungsItem getBestandsWarnungEinzeln(int rezeptur_id) {
    	BestandswarnungsItem ret = null;
    	final String stringLiter = "L";
    	final String stringKg = "Kg";
    	try {
    		if(rezeptur_id != 0){
    			getBestandswarnungEinzeln.setInt(1, rezeptur_id);
    			final ResultSet rs = getBestandswarnungEinzeln.executeQuery();
    			if(rs.next()) {
    				String stringEinheit="";
          			final int eh = rs.getInt("bestandseinheit");
          			if((eh==RzPro.EINHEIT_LITER) || (eh==RzPro.EINHEIT_LA)) {
          				stringEinheit=stringLiter;
          			} else {
          				stringEinheit=stringKg;
          			}
          			ret = new BestandswarnungsItem(
          				rz,
          				rs.getInt("rezeptliste.id"),
          				rs.getDouble("staerke"),
          				eh,
          				stringEinheit,
          				rs.getDouble("diff"),
          				rs.getString("name"),
          				rz.isZero(rs.getDouble("bestand"))?true:false,
          				true,
          				rs.getInt("rezeptliste.id")
          				);
    			}
    		}
    	}catch (final SQLException e){
                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getBestandswarnungEinzeln", e.getLocalizedMessage());
    	}
    	return ret;
    }

    /**
     * Die Liste der Bestandswarnungen holen
     */
    public BestandswarnungsListe getBestandswarnungen(boolean checkBestand,boolean checkBuchungen) {
    	final BestandswarnungsListe bl = new BestandswarnungsListe(rz);
    	final String stringLiter = "L";
    	final String stringKg = "Kg";
    	BestandswarnungsItem bi;
    	try {
    		String exclude = rz.getDatabase().getSTW().dbGetStichwortListe(rz.getOptionFactory().getOption("auswahl.bestandswarnungen.exclude")).toSQLOrString();
      		ResultSet rs = null;
      		if(exclude.trim().equals("")) {
          		rs = getBestandswarnung.executeQuery();
      		} else {
        		getBestandswarnungAuswahl.setString(1, exclude);
          		rs = getBestandswarnungAuswahl.executeQuery();
      		}
;
      		while(rs.next()) {
      			String stringEinheit="";
      			final int eh = rs.getInt("bestandseinheit");
      			if((eh==RzPro.EINHEIT_LITER) || (eh==RzPro.EINHEIT_LA)) {
      				stringEinheit=stringLiter;
      			} else {
      				stringEinheit=stringKg;
      			}
      			bi = new BestandswarnungsItem(
      				rz,
      				rs.getInt("rezeptliste.id"),
      				rs.getDouble("staerke"),
      				eh,
      				stringEinheit,
      				rs.getDouble("diff"),
      				rs.getString("name"),
      				rz.isZero(rs.getDouble("bestand"))?true:false,
      				checkBestand,
      				rs.getInt("rezeptliste.id")
      				);
      			if( rs.getDouble("rest") >0) {
      				bi.setBestellt(true);
      			}
      			bl.addItem(bi);
      		}
    	}
        catch (final SQLException e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getBestandswarnung", e.getLocalizedMessage());
        }
        bl.sortItems();
        if(checkBuchungen) {
        	bl.getDatum();
        }
    	return bl;
    }

    /**
     * Nur die Artikelnummern holen
     */
    public String getArtikelnummer(int id) {
    	String ret = "";
      	try {
      		getLosnummer.setInt(1,id);
      		final ResultSet rs = getLosnummer.executeQuery();
      		if(rs.next()) {
      			ret = rs.getString("artikelnummer");
      		}
    	}
        catch (final SQLException e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getArtikelnummer", e.getLocalizedMessage());
        }
    	return ret;
    }

    /**
     * Nur die Losnummer holen
     */
    public String getLosnummer(int id) {
    	String ret = "";
      	try {
      		getLosnummer.setInt(1,id);
      		final ResultSet rs = getLosnummer.executeQuery();
      		if(rs.next()) {
      			ret = rs.getString("losnummer");
      		}
    	}
        catch (final SQLException e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getLosnummer", e.getLocalizedMessage());
        }
    	return ret;
    }

    /**
     * nur die Stärke holen
     */
    public double getStaerke(int id) {
    	double ret = 0.0;
      	try {
      		getStaerke.setInt(1,id);
      		final ResultSet rs = getStaerke.executeQuery();
      		if(rs.next()) {
      			ret = rs.getDouble("staerke");
      		}
    	}
        catch (final SQLException e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getStaerke", e.getLocalizedMessage());
        }
    	return ret;
    }

    /**
     * nur den typ holen
     */
    public int getTyp(int id) {
    	int ret = 0;
      	try {
      		getTyp.setInt(1,id);
      		final ResultSet rs = getTyp.executeQuery();
      		if(rs.next()) {
      			ret = rs.getInt("typ");
      		}
    	}
        catch (final SQLException e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getTyp", e.getLocalizedMessage());
        }
    	return ret;
    }

    /**
     * nur den raw-comment holen
     */
    public String getComment(int id) {
    	String ret = null;
      	try {
      		getComment.setInt(1,id);
      		final ResultSet rs = getComment.executeQuery();
      		if(rs.next()) {
      			ret = rs.getString("comment");
      		}
    	}
        catch (final SQLException e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getComment", e.getLocalizedMessage());
        }
    	return ret;
    }
    /**
     * Die Liste aller Zutaten einer Rezeptur ermitteln
     */
    private ArrayList getZutaten(ArrayList v) {
    	final ArrayList liste = new ArrayList();
    	try {
        	Statement stmt;
        	String sql="SELECT comment FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE id in(";
        	stmt = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
        	if(!v.isEmpty()) {
	    		for(int i = 0;i < v.size();i++) {
	    			sql += String.valueOf( ((Integer)v.get(i)).intValue()) + ",";
	    		}
	    		if(sql.endsWith(",")) {
					sql = sql.substring(0, sql.length()-1);
				}
	    		sql += ")";
	    		final ResultSet rs = stmt.executeQuery(sql);
		    	while(rs.next()) {
		    		final String zeilen[] = rs.getString("comment").split("\\n");
		    		for (final String element : zeilen) {
		    			if (element.startsWith("1 ")){
		    				try {
		    					final Integer n = Integer.valueOf(Integer.parseInt( element.substring(element.indexOf("[[")+2,element.indexOf("]]"))));
		    					if (liste.indexOf(n) == -1) {
		    						liste.add(n);
		    					}
		    				} catch (final NumberFormatException e) {
		    				}
		    			}
		    		}
				}
        	}
    	}
        catch (final SQLException e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getZutaten", e.getLocalizedMessage());
        }
    	return liste;
    }

    /**
     * Alle Zutaten ermitteln, auf denen eine Rezeptur basiert
     */
    public ArrayList getZutatenListe(int id) {
        final ArrayList el = new ArrayList();
        final ArrayList ret = new ArrayList();
        el.add(Integer.valueOf(id));
        boolean ende= false;
        int count = 0;
        while (!ende) {
       	 final ArrayList al = getZutaten(el);
       	 count++;
       	 for(int i = 0;i < al.size();i++) {
       		 if(ret.indexOf(al.get(i)) == -1) {
       			 try {
       				 ret.add(al.get(i));
       			 }catch(final Exception e) {}
       		 }
       	 }
       	 if ((count > RECURSION_COUNT) || (al.size() == 0)) {
       		 ende = true;
       	 }
       	 el.clear();
       	 el.addAll(al);
        }
       return ret;
    }

    /**
     * Die Deklarationsliste einer Rezeptur zurückgeben
     */
    public String getDeklarationen(int id) {
    	String ret = "";
    	try {
    		getDeklarationen.setInt(1,id);
    		final ResultSet rs = getDeklarationen.executeQuery();
    		if(rs.next()) {
    			ret = rs.getString("deklarationen");
    		}
        }
        catch (final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.getDeklarationen", e.getLocalizedMessage());
        }
        return ret;
    }

    /**
     * eine einzelne Rezeptur automatisch berechnen und die Werte speichern
     */
    public void calcSingle(int id, int typ) {
    	ChangesListe liste = new ChangesListe(rz,id);
    	if(id != 0) {
			switch (typ) {
				case RzPro.TYP_ID_REZEPT:
				case RzPro.TYP_ID_VERWEIS:
					try {
				    	final RechenItem ri = new RechenItem(rz,id,null);
						if(ri.getZeilenListe().isSternzeileError()) {
							 dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
					            dlgItem.setCaption(rz.getLocale().getString("database.rezeptur_update_sternzeile"));
					            dlgItem.setTitel(rz.getLocale().getString("database.rezeptur_update_sternzeile"));
					            dlgItem.setMessage(rz.getLocale().getString("database.error_update_sternzeile_error").replaceAll("%s", ri.getRezeptur().getTitel()));
					            dlgItem.setIcon("frage.png");
					            dlgItem.setOk(rz.getLocale().getString("string_ok"));
					            rz.getDialogFactory().getDialog(dlgItem);
						}
				    	ri.setWaehrung(rz.getDatabase().getCurrency().dbGetCurrency(ri.getRezeptur().getWaehrung(),true));
				    	ri.rechnen(100,RechenItem.RECHNEN_LITER);
						rz.getDatabase().getRezeptur().updateBrennwert(id,ri.getBrennwert(),ri.getBrennwert() - rz.getTafelFactory().Brennwert(ri.getStaerke()));
						rz.getDatabase().getRezeptur().updateExtrakt(ri.getRezeptur().getID(),ri.getExtraktMassenProzent());
						rz.getDatabase().getRezeptur().updateAcid(ri.getRezeptur().getID(),ri.getGesamtAcid());
						rz.getDatabase().getRezeptur().updateDeklarationen(ri.getRezeptur().getID(),ri.getDeklarationenAlle().toString());
						
						
						if(ri.getRezeptur().getExtraktfrei() == true && ri.getExtraktMassenProzent() != 0) {
							 dlgItem.setTyp(DialogControl.FRAGE_DIALOG);
					            dlgItem.setCaption(rz.getLocale().getString("database.rezeptur_update_extraktfrei"));
					            dlgItem.setTitel(rz.getLocale().getString("database.rezeptur_update_extraktfrei"));
					            dlgItem.setMessage(rz.getLocale().getString("database.error_update_extraktfrei")
					            			.replaceAll("%name",ri.getRezeptur().getTitel())
					            			.replaceAll("%extrakt", rz.getLocale().formatNumber(ri.getExtraktMassenProzent(), OptionFactory.NF_NORMAL))
					            			);
					            dlgItem.setIcon("frage.png");
					            dlgItem.setOk(rz.getLocale().getString("string_ja"));
					            dlgItem.setCancel(rz.getLocale().getString("string_nein"));
					            rz.getDialogFactory().getDialog(dlgItem);
					            if((dlgItem.getReplyCode() == DialogControl.DIALOG_OK) || (dlgItem.getReplyCode()==DialogControl.DIALOG_YES)) {
					            	ri.getRezeptur().setExtraktfrei(false);
					            	rz.getDatabase().getRezeptur().updateExtraktfrei(id,0);
					            }
						}
						
						if(ri.getRezeptur().getExtraktfrei() == false && rz.isZero(ri.getExtraktMassenProzent())) {
							 dlgItem.setTyp(DialogControl.FRAGE_DIALOG);
					            dlgItem.setCaption(rz.getLocale().getString("database.rezeptur_update_extraktfrei"));
					            dlgItem.setTitel(rz.getLocale().getString("database.rezeptur_update_extraktfrei"));
					            dlgItem.setMessage(rz.getLocale().getString("database.error_update_extraktfrei_1")
					            			.replaceAll("%name",ri.getRezeptur().getTitel())
					            			.replaceAll("%extrakt", rz.getLocale().formatNumber(ri.getExtraktMassenProzent(), OptionFactory.NF_NORMAL))
					            			);
					            dlgItem.setIcon("frage.png");
					            dlgItem.setOk(rz.getLocale().getString("string_ja"));
					            dlgItem.setCancel(rz.getLocale().getString("string_nein"));
					            rz.getDialogFactory().getDialog(dlgItem);
					            if((dlgItem.getReplyCode() == DialogControl.DIALOG_OK) || (dlgItem.getReplyCode()==DialogControl.DIALOG_YES)) {
					            	ri.getRezeptur().setExtraktfrei(false);
					            	rz.getDatabase().getRezeptur().updateExtraktfrei(id,1);
					            }
						}
						
						
						if(ri.getRezeptur().getExtraktfrei() == false) {
							rz.getDatabase().getRezeptur().updateLitergewicht(ri.getRezeptur().getID(),ri.getGesamtLitergewicht());
						}
						double preisfaktor = 0;
						switch(ri.getRezeptur().getPreisEinheit()) {
							case RzPro.EINHEIT_KG:
								preisfaktor = ri.getRezeptur().getPreisMenge() / ri.getGesamtKg();
								break;
							case RzPro.EINHEIT_LA:
								preisfaktor = ri.getRezeptur().getPreisMenge() / ri.getGesamtLA();
								break;
							case RzPro.EINHEIT_LITER:
								preisfaktor = ri.getRezeptur().getPreisMenge()/ri.getGesamtLiter();
								break;
						}
						BigDecimal preis = ri.getSummeOhneSteuer().multiply(new BigDecimal(preisfaktor)).setScale(RzPro.CURRENCY_DEFAULT_SCALE,BigDecimal.ROUND_HALF_UP);
						preis = rz.getDatabase().getCurrency().convert(preis,ri.getWaehrung(),rz.getDatabase().getCurrency().dbGetCurrency(ri.getRezeptur().getWaehrung(),true));
						rz.getDatabase().getRezeptur().updatePreis(ri.getRezeptur().getID(),preis);
						} catch(final Exception e) {
						rz.getLogFactory().logMessage(LogFactory.LOG_WARN,"auswahl.calcSingle",e.getLocalizedMessage());
					}
					break;
				case RzPro.TYP_ID_AUSSTATTUNG:
					final AusstattungsItem ai = db.getAusstattung().getAusstattung(id);
					if(ai != null) {
						rz.getDatabase().getAusstattung().updatePreis(ai);
					}
					break;
			}
    	}
    	if(liste != null) {
    		liste.commit();
    	}
    }

        /**---------------------------------------------------------------------
     *  Funktionen für Rezepturen
     *----------------------------------------------------------------------
     */
    /**
     * Brennwert einer Rezeptur updaten
     * @param id = ID der Rezeptur
     * @param hinweis
     *
     */
    public void updateBrennwert(int id,double brennwert,double brennwert_af)   {
    	try {
    		updateBrennwert.setDouble(1,brennwert);
    		updateBrennwert.setDouble(2,brennwert_af);
    		updateBrennwert.setInt(3,id);
    		updateBrennwert.executeUpdate();
    	}catch (final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.updateBrennwert", e.getLocalizedMessage());
    	}
    }

    /**
     * Hinweis einer Rezeptur updaten
     * @param id = ID der Rezeptur
     * @param hinweis
     *
     */
    public void updateHinweis(int id,String hinweis)   {
    	try {
    		updateHinweis.setString(1,hinweis);
    		updateHinweis.setString(2,db.dbGetUser());
    		updateHinweis.setString(3,db.dbGetDateFormatString(db.getServerTimestamp()));
    		updateHinweis.setInt(4,id);
    		updateHinweis.executeUpdate();
    	}catch (final SQLException e) {
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezepturen.updateHinweis", e.getLocalizedMessage());
    	}
    }
    /**
     * holt nur die Zahl für den Bestand aus der Datenbank
     * @param id   Rezeptur-ID
     */
    public double getBestand(int id) {
        double ret = 0.0;
        if (id != 0){
            try{
            	final SaldenItem s = rz.getDatabase().getRezeptur().getBestandsSaldo(id);
            	if(s.getBestandseinheit() == RzPro.EINHEIT_LITER) {
            		ret = s.getLiter();
            	} else if (s.getBestandseinheit() == RzPro.EINHEIT_KG){
            		ret = s.getKg();
            	} else if(s.getBestandseinheit() == RzPro.EINHEIT_LA) {
            		ret = s.getLa();
            	}
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.getBestand", e.getLocalizedMessage());
               ret = 0.0;
            }
        }
        return ret;
    }

    /**
     * holt den Bestand als SaldenItem aus den Buchungen einer Rezeptur
     * @param id
     * @param bewertung
     * @return
     */
    public SaldenItem getBestandsSaldo(int id) {
    	final SaldenItem ret = new SaldenItem(0,0,0,RzPro.EINHEIT_LITER);
        if (id != 0){
            try{
            	getBestandsSaldo.setInt(1,id);
            	final ResultSet rs = getBestandsSaldo.executeQuery();
            	if(rs.next()) {
            		ret.setLiter(rs.getDouble("rest_liter"));
            		ret.setKg(rs.getDouble("rest_kg"));
            		ret.setLa(rs.getDouble("rest_la"));
            		ret.setBestandseinheit(rs.getInt("bestandseinheit"));
            	} else {
            		ret.setBestandseinheit(getBestandseinheit(id));
            	}
            	
            	rs.close();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.getBestand", e.getLocalizedMessage());
               ret.setLiter(0);
               ret.setKg(0);
               ret.setLa(0);
               ret.setBestandseinheit(RzPro.EINHEIT_LITER);
            }
        }
    	return ret;
    }

    public int getBestandseinheit(int id) {
    	int ret = RzPro.EINHEIT_LITER;
        if (id != 0){
            try{
            	getBestandseinheit.setInt(1,id);
            	final ResultSet rs = getBestandseinheit.executeQuery();
            	if(rs.next()) {
            		ret = rs.getInt("bestandseinheit");
            	} 
            	rs.close();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.getBestand", e.getLocalizedMessage());
            }
        }
    	return ret;
    }
    

    public boolean updateBestand(int id,double value) {
        boolean ret = true;
        if (id > 0){
            try{
            	if(value <0) {
					value=0.0;
				}
            	if(rz.isZero(value)) {
					value = 0.0;
				}
                updateBestand.setDouble(1,value);
                updateBestand.setInt(2,id);
                final int r = updateBestand.executeUpdate();
                if(r == 0) {
                    ret = false;
                }
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateBestand", e.getLocalizedMessage());
               ret = false;
            }
        }
        return ret;
    }

    public SaldenItem  forceBestand(int id) {
    	SaldenItem s = new SaldenItem(0,0,0,RzPro.EINHEIT_LITER);
        if (id > 0){
        	s = rz.getDatabase().getRezeptur().getBestandsSaldo(id);
        	double value = s.getBestand();
        	try{
            	if(value <0) {
					value=0.0;
				}
            	if(rz.isZero(value)) {
					value = 0.0;
				}
                updateBestand.setDouble(1,value);
                updateBestand.setInt(2,id);
                updateBestand.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.forceBestand", e.getLocalizedMessage());
            }
        }
        return s;
    }

    public SaldenItem  getBestandStichtag(int id,Calendar d) {
    	final SaldenItem ret = new SaldenItem(0,0,0,RzPro.EINHEIT_LITER);
        if (id != 0){
            try{
            	String datum =	d.get(Calendar.YEAR)+"-"+(d.get(Calendar.MONTH)+1)+"-"+d.get(Calendar.DAY_OF_MONTH) + " 23:59:59";
            	getBestandsSaldoDatum.setInt(1,id);
            	getBestandsSaldoDatum.setString(2,datum);
            	final ResultSet rs = getBestandsSaldoDatum.executeQuery();
            	if(rs.next()) {
            		ret.setLiter(rs.getDouble("rest_liter"));
            		ret.setKg(rs.getDouble("rest_kg"));
            		ret.setLa(rs.getDouble("rest_la"));
            		ret.setBestandseinheit(rs.getInt("bestandseinheit"));
            	} else {
            		ret.setBestandseinheit(getBestandseinheit(id));
            	}
            	
            	rs.close();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.getBestandDatum", e.getLocalizedMessage());
               ret.setLiter(0);
               ret.setKg(0);
               ret.setLa(0);
               ret.setBestandseinheit(RzPro.EINHEIT_LITER);
            }
        }
    	return ret;
    }
    
    public void updateGeaendert(int id) {
        if (id > 0){
            try{
                updateGeaendert.setString(1,db.dbGetDateFormatString(db.getServerTimestamp()));
                updateGeaendert.setString(2,db.dbGetUser());
                updateGeaendert.setInt(3,id);
                updateGeaendert.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateExtraktfrei", e.getLocalizedMessage());
            }
        }
    }
    
    public void updateText(int id,double staerke,String raw) {
        if (id > 0){
            try{
                updateText.setString(1,raw);
                updateText.setDouble(2,staerke);
                updateText.setString(3,db.dbGetUser());
                updateText.setString(4,db.dbGetDateFormatString(db.getServerTimestamp()));
                updateText.setInt(5,id);
                updateText.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateExtrakt", e.getLocalizedMessage());
            }
        }
    }

    public void updateKalkulation(int id,double preismenge,int einheit,BigDecimal preis,int waehrung) {
        if (id > 0){
            try{
                updateKalkulation.setBigDecimal	(1,preis);
                updateKalkulation.setInt(2,einheit);
                updateKalkulation.setDouble(3,preismenge);
                updateKalkulation.setInt(4,waehrung);
                updateKalkulation.setInt(5,id);
                updateKalkulation.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateExtrakt", e.getLocalizedMessage());
            }
        }
    }

    public void updateStaerke(int id,double staerke) {
        if (id > 0){
            try{
                updateStaerke.setDouble(1,staerke);
                updateStaerke.setInt(2,id);
                updateStaerke.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateExtrakt", e.getLocalizedMessage());
            }
        }
    }
    
    
    public void updateExtrakt(int id,double extrakt) {
        if (id > 0){
            try{
                updateExtrakt.setDouble(1,extrakt);
                updateExtrakt.setInt(2,id);
                updateExtrakt.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateExtrakt", e.getLocalizedMessage());
            }
        }
    }

    public void updateExtraktfrei(int id,int value) {
        if (id > 0){
            try{
                updateExtraktfrei.setInt(1,value);
                updateExtraktfrei.setInt(2,id);
                updateExtraktfrei.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateExtraktfrei", e.getLocalizedMessage());
            }
        }
    }
    
    public void updateAcid(int id,double acid) {
        if (id > 0){
            try{
                updateAcid.setDouble(1,acid);
                updateAcid.setInt(2,id);
                updateAcid.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateAcid", e.getLocalizedMessage());
            }
        }
    }

    public void updateDeklarationen(int id,String deklarationen_alle) {
        if (id > 0){
            try{
                updateDeklarationen.setString(1,deklarationen_alle);
                updateDeklarationen.setInt(2,id);
                updateDeklarationen.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateDeklarationen", e.getLocalizedMessage());
            }
        }
    }

    public void updatePreis(int id,BigDecimal preis) {
        if (id > 0){
            try{
                updatePreis.setBigDecimal(1,preis);
                updatePreis.setInt(2,id);
                updatePreis.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updatePreis", e.getLocalizedMessage());
            }
        }
    }
    public void updateLitergewicht(int id,double literGewicht) {
        if (id > 0){
            try{
            	if(literGewicht <=0) {
					literGewicht = rz.getTafelFactory().Litergewicht(0);
				}
                updateLitergewicht.setDouble(1,literGewicht);
                updateLitergewicht.setInt(2,id);
                updateLitergewicht.executeUpdate();
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.updateLitergewicht", e.getLocalizedMessage());
            }
        }
    }

     /** int getRezepturID(String) liefert die ID zu einem Namen
      */

    public int dbGetRezepturID(String aName){
        int ret = 0;
        ResultSet rs;
        if (aName != null){
            try{
                getRezepturID.setString(1,aName.trim());
                rs = getRezepturID.executeQuery();
                if(rs.next()){
                    ret = rs.getInt("id");
                }
                if(rs != null) {
					rs.close();
				}
            }
            catch(final Exception e){
               rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbGetRezepturId", e.getLocalizedMessage());
            }
        }
        return ret;
    }
    /** String dbGetRezeptur(int) holt nur den Namen einer Rezeptur
     */
    public String dbGetRezepturName(int aKey) throws ItemNotFoundException{
        String ret = "";
        ResultSet rs;
        try{
            getRezepturName.setInt(1,aKey);
            rs = getRezepturName.executeQuery();
            if(rs.next()){
                ret = rs.getString("name");
            } else {
                throw new ItemNotFoundException();
            }
            if(rs != null) {
				rs.close();
			}
        }
        catch(final Exception e){
           rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbGetRezepturName", e.getLocalizedMessage());
        }
        return ret;
    }

    
    
    
    public String dbGetRezepturNameNEX(int aKey){
        String ret = "";
        ResultSet rs;
        try{
            getRezepturName.setInt(1,aKey);
            rs = getRezepturName.executeQuery();
            if(rs.next()){
                ret = rs.getString("name");
            }
			rs.close();
        }
        catch(final Exception e){
           rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbGetRezepturName", e.getLocalizedMessage());
        }
        return ret;
    }

    /**
     * dbGetRezeptur(int) holt eine bestimmte Rezeptur aus der Datenbank
     * oder initialisiert eine neue RezepturStruktur wenn der übergebene
     * Key==0 ist und gibt diese zurück
     *
     */


    public RezepturItem dbGetRezeptur(int aKey,boolean quiet){
        RezepturItem ret = null;
        ResultSet rs;
        if((aKey == 0) || (aKey == REZEPTURWASSER)){
           ret = null;
           ret = new RezepturItem(
                rz,
                0,
                RzPro.TYP_ID_REZEPT,
                (aKey == REZEPTURWASSER) ? rz.getLocale().getString("database.verschnittwasser" ) :rz.getLocale().getString("database.neue_rezeptur"),
				0,
				"",
                new StichwortListe(),
                new StichwortListe(),
                new DeklarationsListe(rz),
                new DeklarationsListe(rz),
                0,
                true,
                0,
                0.998201,
                1,
                100,//Basismenge
                0,
                0,
                0,
                1,
                "",
                "",
                new BigDecimal(0.0),
                1, //preis
                1, //preismenge
                1, //preiseinheit
                1, //währung
                0,
                db.getServerTimestamp(),
                db.getServerTimestamp(),
                db.dbGetUser(),
                db.dbGetUser(),
                "",
                "",
                db.getServerTimestamp(),
                0,
                0,
                0,
                "",
                "",
                (rz.getOptionFactory().getOption("einstellungen.probe_erforderlich",0)==0)?false:true,
                20,
                20,
                null,
                "",
                false,
                "",
                false,
                "",
                	"",
                	1,
                	1,
                	false,
                	1
           );
        }
        else {
            try{
            	if(db.idExists(DBFactory.TABLE_REZEPTLISTE,aKey,quiet)) {
            		getRezeptur.setInt(1, aKey);
            		rs = getRezeptur.executeQuery();
                	final SaldenItem s = rz.getDatabase().getRezeptur().getBestandsSaldo(aKey);
	                if(rs.next()){
	                	double bestand=0.0;
	                	int bestandseinheit = s.getBestandseinheit();
	                	if(bestandseinheit == RzPro.EINHEIT_LITER) {
	                		bestand = s.getLiter();
	                	} else if (bestandseinheit == RzPro.EINHEIT_KG){
	                		bestand=s.getKg();
	                	} else if(bestandseinheit == RzPro.EINHEIT_LA) {
	                		bestand=s.getLa();
	                	}
	                    ret = new RezepturItem(
	                    rz,
	                    aKey,
	                    rs.getInt("typ"),
	                    rs.getString("name"),
						rs.getInt("verweis"),
	                    rs.getString("comment"),
	                    db.getSTW().dbGetAdressenListe(rs.getString("adressen")),
	                    db.getSTW().dbGetStichwortListe(rs.getString("stichworte")),
	                    db.getDeklarationen().getDeklarationenListe(rs.getString("deklarationen")),
	                    db.getDeklarationen().getDeklarationenListe(rs.getString("deklarationen_alle")),
	                    rs.getDouble("staerke"),
	                    rs.getBoolean("extraktfrei"),
	                    rs.getDouble("extrakt"),
	                    rs.getDouble("litergewicht"),
	                    rs.getInt("basiseinheit"),
	                    rs.getDouble("basismenge"),
	                    bestand,
	                    rs.getDouble("meldebestand"),
	                    rs.getDouble("vorschlagsbestand"),
	                    bestandseinheit,
	                    rs.getString("artikelnummer"),
	                    rs.getString("losnummer"),
	                    rs.getBigDecimal("preis"),
	                    rs.getDouble("preismenge"),
	                    rs.getInt("preiseinheit"),
	                    rs.getInt("waehrung"),
	                    rs.getInt("lager"),
	                    rs.getDouble("schwundsatz"),
	                    db.getDateFromString(rs.getString("erstellt")),
	                    db.getDateFromString(rs.getString("geaendert")),
	                    rs.getString("user_1"),
	                    rs.getString("user_2"),
	                    rs.getString("hinweis"),
	                    rs.getString("hinweis_user"),
	                    db.getDateFromString(rs.getString("hinweis_datum")),
	                    rs.getDouble("brennwert"),
	                    rs.getDouble("brennwert_af"),
	                    rs.getDouble("acid"),
	                    rs.getString("tank"),
	                    rs.getString("spezifikation"),
	                    rs.getInt("r_probe")==0?false:true,
	                    rs.getInt("tempvon"),
	                    rs.getInt("tempbis"),
	                    (rs.getString("konformDatum") != null) ?db.getDateFromString(rs.getString("konformDatum")):null,
	                    rs.getString("konformComment"),
	                    rs.getInt("konform") != 0 ? true:false,
	                    rs.getString("spezifikation_text"),
	                    rs.getInt("bewertung") != 0 ? true:false,
	               		rs.getString("ean"),
	               		rs.getString("steuer_artikel"),
	                    rs.getInt("sorte"),
	                    rs.getInt("steuerlager"),
	                    rs.getInt("auto_anfrage") == 1?true:false,
	                    rs.getInt("proben_anzahl")
	                    );
	                }
	                if(rs != null) {
						rs.close();
					}
            	}
            }
            catch (final SQLException e){
                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbGetRezeptur", e.getLocalizedMessage() + ":" + aKey);
                ret = null;
            }
        }
        return ret;
    }
    /**
     * getVerweise gibt eine Liste der Namen zurück, die auf diese Rezeptur verweisen
     * @param aID id der REzeptur
     * @return String verweisliste;
     */
    public String dbGetVerweisListe(int aID) {
        String ret = "";
        ResultSet rs;
        try{
        	if(aID != 0) {
	            getVerweise.setInt(1,aID);
	            rs = getVerweise.executeQuery();
	            while (rs.next()){
	                ret += rs.getString("name") + ",";
	            }
        	}
        }
        catch(final Exception e){
           rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbGetVerweisListe", e.getLocalizedMessage());
        }
        if(!ret.equals("")) {
			ret = ret.substring(0,ret.length()-1);
		}
        return ret;
    }
    /**
     * SaveVerweis speichert einen Verweis
     * @param ri RezepturItem
     * @return Id des eingefügten/upgedateten Datensatzes
     */
    public int dbSaveVerweis(RezepturItem ri) {
        int ret = 0;
        PreparedStatement stm;
        final RezepturItem source = dbGetRezeptur(ri.getVerweis(),true);
        if(ri.CommentToRaw(ri.getComment())) {
	        try {
	            if (ri.getID() == 0) {
					stm = saveVerweis;
				} else {
	                stm = updateVerweis;
	                stm.setInt(10, ri.getID());
	                ri.setUser_2(db.user);
	                ri.setGeaendert(db.getServerTimestamp());
	            }
	            // verweis sichern
	            stm.setInt(1,RzPro.TYP_ID_VERWEIS);
	            stm.setString(2,  ri.getTitel());
	            stm.setInt(3,ri.getVerweis());
	            stm.setString(4, ri.getAdressListe().toString());
	            stm.setString(5, ri.getStichwortListe().toString());
	            stm.setString(6,ri.getUser1());
	            stm.setString(7,ri.getUser2());
	            stm.setString(8,db.dbGetDateFormatString(ri.getErstellt()));
	            stm.setString(9,db.dbGetDateFormatString(ri.getGeaendert()));
	            if(stm.executeUpdate() != 0){
	                if(ri.getID() == 0){
	                    ret = db.dbGetLastInsertID("" + DBFactory.TABLE_REZEPTLISTE + "");
	                }
	                else {
	                    ret = ri.getID();
	                }
	            }
	            else{
	                rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbSaveVerweis:","Update Row " + ri.getID() + " not found, creating new Row !");
	                ri.setID(0);
	                ret = dbSaveVerweis(ri);
	            }
	            //source sichern
	            ri.setID(source.getID());
	            ri.setTitel(source.getTitel());
	            ri.setAdressListe(source.getAdressListe());
	            ri.setStichwortListe(source.getStichwortListe());
	            ri.setErstellt(source.getErstellt());
	            ri.setGeaendert(source.getGeaendert());
	            ri.setUser_1(source.getUser1());
	            ri.setUser_2(source.getUser2());
	            ri.setVerweis(0);
	            dbSaveRezeptur(ri);
	        }
	        catch (final Exception e){
	            ret = -1;
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbSaveVerweis", e.getLocalizedMessage());
	        }
	     }
        return ret;
    }

    /**
     * dbSaveRezeptur(RezepturItem) speichert eine Rezeptur in der Datenbank oder macht
     */
    public int dbSaveRezeptur(RezepturItem ri){
        int ret = 0;
        ChangesListe liste = new ChangesListe(rz,ri.getID());
        boolean listCommit=ri.getID()==0?false:true;
        final Date datum = db.getServerTimestamp();
        PreparedStatement stm;
        if(ri.CommentToRaw(ri.getComment())) {
	        try {
	            if (ri.getID() == 0) {
	                stm = saveRezeptur;
	            	stm.setString(28, db.dbGetDateFormatString( datum));
	            	stm.setString(29, db.dbGetDateFormatString(datum ));
	            }
	            else {
	                stm = updateRezeptur;
		            stm.setString(28, db.dbGetDateFormatString( ri.getErstellt()));
		            stm.setString(29, db.dbGetDateFormatString( datum ));
		            ri.setUser_2(db.dbGetUser());
	                stm.setInt(51, ri.getID());
	                if(!db.idExists(DBFactory.TABLE_REZEPTLISTE,ri.getID(),true)) {
	                	stm=saveRezeptur;
	                	ri.setID(0);
	                }
	            }
	            double bestand=0.0;
	            SaldenItem si = this.getBestandsSaldo(ri.getID());
	            switch(ri.getBestandseinheit()) {
	            	case RzPro.EINHEIT_LITER: bestand=si.getLiter();
	            		break;
	            	case RzPro.EINHEIT_LA:bestand =si.getLa();
	            		break;
	            	case RzPro.EINHEIT_KG:bestand=si.getKg();
	            	break;
	            }
	            stm.setInt(1,ri.getVerweis() == 0 ?RzPro.TYP_ID_REZEPT : RzPro.TYP_ID_VERWEIS);
	            stm.setString(2,  ri.getTitel());
	            stm.setInt(3,ri.getVerweis());
	            stm.setString(4, ri.getRaw());
	            stm.setString(5, ri.getAdressListe().toString());
	            stm.setString(6, ri.getStichwortListe().toString());
	            stm.setString(7,ri.getDeklarationenListe().toString());
	            stm.setString(8,ri.getDeklarationenAlle().toString());
	            stm.setDouble(9, ri.getStaerke());
	            stm.setInt(10, ri.getExtraktfrei() == true ? 1 : 0);
	            stm.setDouble(11, ri.getExtrakt());
	            stm.setDouble(12, ri.getLitergewicht());
	            stm.setInt(13, ri.getBasiseinheit());
	            stm.setDouble(14, ri.getBasismenge());
	            stm.setDouble(15, bestand);
	            stm.setDouble(16, ri.getMeldebestand());
	            stm.setDouble(17,ri.getVorschlagsbestand());
	            stm.setInt(18, ri.getBestandseinheit());
	            stm.setString(19, ri.getArtikelnummer());
	            stm.setString(20, ri.getLosnummer());
	            stm.setBigDecimal(21, ri.getPreis());
	            stm.setDouble(22, ri.getPreisMenge());
	            stm.setInt(23, ri.getPreisEinheit());
	            stm.setInt(24, ri.getWaehrung());
	            stm.setInt(25, ri.getLager());
	            stm.setString(26,ri.getHinweis());
	            stm.setDouble(27, ri.getSchwundsatz());
	            // erstellt-geändert wird oben gesetzt
	            stm.setString(30, ri.getUser1());
	            stm.setString(31, ri.getUser2());
	            stm.setDouble(32,ri.getBrennwert());
	            stm.setDouble(33,ri.getBrennwertAF());
	            stm.setDouble(34,ri.getAcid());
	            stm.setString(35,ri.getTank());
	            stm.setString(36,ri.getSpezifikation());
	            stm.setInt(37, ri.isRProbe()==false?0:1);
	            stm.setInt(38, ri.getTempVon());
	            stm.setInt(39, ri.getTempBis());
	            stm.setString(40, ri.getKonformDatum()!=null?db.dbGetDateFormatString(ri.getKonformDatum()):null);
	            stm.setString(41, ri.getKonformComment());
	            stm.setInt(42, ri.isKonform()?1:0);
	            stm.setString(43, ri.getSpezifikation_text()==null?"":ri.getSpezifikation_text());
	            stm.setInt(44, ri.isBewertung()?1:0);
            		stm.setString(45, ri.getEan());
            		stm.setString(46, ri.getSteuerArtikel());
            		stm.setInt(47, ri.getSorte());
            		stm.setInt(48, ri.getSteuerLager());
            		stm.setInt(49, ri.isAuto_anfrage()?1:0);
            		stm.setInt(50, ri.getProbenAnzahl());
	            if(stm.executeUpdate() != 0) {
	                if(ri.getID() == 0){
	                    ret = db.dbGetLastInsertID("" + DBFactory.TABLE_REZEPTLISTE + "");
	                }
	                else {
	                    ret = ri.getID();
	                }
	            }
	        }
	        catch (final Exception e){
	            ret = -1;
	            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbSaveRezeptur", e.getLocalizedMessage());
	        }
        }
        if(liste != null && listCommit) {
        	liste.commit();
        }
        return ret;
    }

    /** dbCheckRezepturUsed prüft ob eine Rezeptur in der Datenbank gebraucht wird..
     * geprüft wird die rezeptliste selbst, ob die Rezeptur evtl. als Zutat verwendet wird und
     * die Buchungsliste, ob die Rezeptur schon einmal verbucht wurde
     * @param aID zu überprüfende id
     * @param aName Name der Zutat
     *
     * @return true, wenn nicht gelöscht werden kann, false wenn gelöscht werden kann
     */
    public boolean dbCheckRezepturUsed(int aID,String aName){
        boolean ret = true;
        ResultSet rs;
        try{
            //Pass 1... prüfen, ob die Rezeptur irgendwo als Zutat verwendet wird
            checkRezepturRezeptliste.setString(1,"\\[\\[" + aID+ "\\]\\]");
            rs = checkRezepturRezeptliste.executeQuery();
            if(rs.next()){
                // rezeptur wird gebraucht und kann nicht gelöscht werden
                dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
                dlgItem.setCaption(rz.getLocale().getString("database.rezeptur_loeschen"));
                dlgItem.setTitel(rz.getLocale().getString("database.rezeptur_loeschen"));
                dlgItem.setMessage(rz.getLocale().getString("database.error_rezeptur_wird_verwendet").replaceAll("%s",aName));
                dlgItem.setIcon("warnung.png");
                dlgItem.setOk(rz.getLocale().getString("string_ok"));
                rz.getDialogFactory().getDialog(dlgItem);
                return true;
            }
            else {
                ret = false;
            }

            //Pass 2... prüfen ob Verweise auf die Rezeptur existieren
            checkVerweis.setInt(1,aID);
            rs = checkVerweis.executeQuery();
            if(rs.next()){
                // rezeptur hat Verweise und kann nicht gelöscht werden
                dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
                dlgItem.setCaption(rz.getLocale().getString("database.rezeptur_loeschen"));
                dlgItem.setTitel(rz.getLocale().getString("database.rezeptur_loeschen"));
                dlgItem.setMessage(rz.getLocale().getString("database.error_rezeptur_hat_verweis").replaceAll("%s",aName));
                dlgItem.setIcon("warnung.png");
                dlgItem.setOk(rz.getLocale().getString("string_ok"));
                rz.getDialogFactory().getDialog(dlgItem);
                return true;
            }
            else {
                ret = false;
            }
            //Pass 3... prüfen ob die Rezeptur schonmal verbucht wurde
            if(db.getBuchungen().checkRezepturUsed(aID)){
                // rezeptur hat Verweise und kann nicht gelöscht werden
                dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
                dlgItem.setCaption(rz.getLocale().getString("database.rezeptur_loeschen"));
                dlgItem.setTitel(rz.getLocale().getString("database.rezeptur_loeschen"));
                dlgItem.setMessage(rz.getLocale().getString("database.error_rezeptur_hat_buchungen").replaceAll("%s",aName));
                dlgItem.setIcon("warnung.png");
                dlgItem.setOk(rz.getLocale().getString("string_ok"));
                rz.getDialogFactory().getDialog(dlgItem);
                return true;
            }
            else {
                ret = false;
            }

            if(rs != null) {
				rs.close();
			}
        }
        catch (final Exception e){
           rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbCheckRezeptur:", e.getLocalizedMessage());
        }
        return ret;
    }
    /** dbDeleteRezeptur(int) löscht eine Rezeptur aus der Datenbank
     * prüft _nicht_ ob die Rezeptur gelöscht werden kann !!!
     * @see dbCheckRezepturUsed
     */
    public int dbDeleteRezeptur(int aID){
        final int ret = 0;
        try {
            deleteRezeptur.setInt(1,aID);
            if (deleteRezeptur.executeUpdate() == 0){
                dlgItem.setTyp(DialogControl.HINWEIS_DIALOG);
                dlgItem.setCaption(rz.getLocale().getString("rezeptur.key_not_found"));
                dlgItem.setTitel(rz.getLocale().getString("rezeptur.key_not_found"));
                dlgItem.setMessage(rz.getLocale().getString("rezeptur.key_not_found"));
                dlgItem.setIcon("warnung.png");
                dlgItem.setOk(rz.getLocale().getString("string_ok"));
                rz.getDialogFactory().getDialog(dlgItem);
            } else {
            	deleteVersionen(aID);
            }
        }
        catch(final Exception e){
        	rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbDeleteRezeptur:", e.getLocalizedMessage());
        }
        return ret;
    }
    /** gibt die Liste aller Zutaten zurück, die mit einem bestimmten String anfangen
     *
     * @author labor
     *
     */

    /** dupliziert eine Rezeptur und ändert den Namen der Kopie
     *
     */
    public int dbDuplicateRezeptur(int aKey,String aName) {
    	int ret = 0;
    	if(!aName.equals("")) {
	    	final RezepturItem ri = dbGetRezeptur(aKey,false);
	    		if(ri != null) {
	    		ri.setID(0);
	    		ri.setTitel(aName);
	    		ri.setGeaendert(db.getServerTimestamp());
	    		ri.setUser_2(db.user);
	    		ri.setHinweis("");
	    		ri.setAuto_anfrage(false);
	    		ret = dbSaveRezeptur(ri);
    		}
    	}
    	return ret;
    }
    /** gibt eine Liste alle Zutaten zurück, bei dem die Zutaten mit dem
     * @param aVorgabe anfangen
     * @return Zutatenliste
     */
    public ZutatenListe dbGetZutatenListe(String aVorgabe,int mode,boolean prefix){
    	final ZutatenListe zl = new ZutatenListe(rz);
    	StichwortListe stw = rz.getDatabase().getSTW().getAbgangsClientFilter();
    	boolean insert = false;
    	int limit =  rz.getOptionFactory().getOption("zutatendialog.limit", ZutatenDialog.DEFAULT_LIMIT);
   	   ResultSet rs;
    	String ablage=rz.getDatabase().getErgebnis().getAblageFilter();
    	try{
    		if(!aVorgabe.trim().equals("")) {
    			PreparedStatement stm = null;
    			switch(mode) {
    				case ZutatenDialog.MODE_AUSSTATTUNG :
    					stm = getZutatenListeAusstattung;
    		    		stm.setInt(2,limit);
    				break;
    				case ZutatenDialog.MODE_ALKOHOLHALTIG:
    				case ZutatenDialog.MODE_EXTRAKTHALTIG:    					
    				case ZutatenDialog.MODE_REZEPTUREN:
    					if(ablage.equals("")) {
    						stm = getZutatenListe;
    		    			stm.setInt(2,limit);
    					} else {
    						stm = getZutatenListeAblage;
    						stm.setString(2, ablage);
    		    			stm.setInt(3,limit);
    					}
    					break;
    				case ZutatenDialog.MODE_FERTIGWAREN_AUSWAHL:
    					if(ablage.equals("")){
    							stm = getZutatenListeFertigwaren;
    							stm.setString(2,stw.toSQLOrString());
    							stm.setInt(3,limit);
    					} else {
							stm = getZutatenListeFertigwarenAblage;
							stm.setString(2,stw.toSQLOrString());
							stm.setString(3, ablage);
							stm.setInt(4,limit);	
    					}
    					break;
    			}
    			if(rz.isMain()) {
    				if(prefix) {
    					stm.setString(1,rz.getOptionFactory().getOption("zutatendialog.prefix") + aVorgabe.replaceAll("\\s",".*"));
    				} else {
    					stm.setString(1,aVorgabe.replaceAll("\\s",".*"));    					
    				}
    			}else {
    				stm.setString(1,aVorgabe.replaceAll("\\s",".*"));
    			}
				rs = stm.executeQuery();
				while (rs.next()){
				    switch(mode) {
				    	case ZutatenDialog.MODE_AUSSTATTUNG:
				    	case ZutatenDialog.MODE_FERTIGWAREN_AUSWAHL:
				    	case ZutatenDialog.MODE_REZEPTUREN : insert = true; break;
				    	case ZutatenDialog.MODE_ALKOHOLHALTIG :
				    	    if(rz.isZero(rs.getDouble("staerke"))){
				    	        insert = false;
				    	    } else {
				    	        insert = true;
				    	    }
				    	break;
				    	case ZutatenDialog.MODE_EXTRAKTHALTIG :
				    	    if(rz.isZero(rs.getDouble("extrakt"))){
				    	        insert = false;
				    	    } else {
				    	        insert = true;
				    	    }
				    	break;
				    }
				    if(insert) {
				    	StichwortListe l = db.getSTW().dbGetStichwortListe(rs.getString("stichworte"));
				    	zl.addItem(new ZutatenSuchItem(
								rs.getInt("id"),
								rs.getString("name"),
								rs.getString("artikelnummer"),
								rs.getDouble("staerke"),
								rs.getDouble("extrakt"),
								rs.getDouble("litergewicht"),
								l,
								rz.isZero(rs.getDouble("bestand"))?false:true,
								rz.isZero(rs.getDouble("meldebestand"))?false:rs.getDouble("bestand") < rs.getDouble("meldebestand")?true:false
						));
					}
				}
				if(rs != null) {
					rs.close();
				}
    		}
    	}
        catch(final Exception e){
            rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.dbGetZutatenListe:", e.getLocalizedMessage());
       }
    	zl.setExclusive();
    	return zl;
    }

    /** füllt eine Combobox mit den extrakthaltigen Zutaten einer Rezeptur
     *
     * @param cmb
     * @param buchung_id
     */
	 public void setExtraktCmb(JComboBox cmb,String vorgabe,RezepturItem ri){
	     int i;
	     cmb.removeAllItems();
	     final String zeilen[] = ri.getRaw().split("\\n");
	     String sql = "SELECT DISTINCT id,staerke FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE ";
	     if(!vorgabe.equals("")) {
	         final String keys[] = vorgabe.split(";");
	         for(i=0;i < keys.length;i++) {
	             sql += " id=" + keys[i] + " OR ";
	     	}
	     }
	     for( i = 0 ; i <zeilen.length;i++){
	         if(zeilen[i].startsWith("1 ")) {
	             	 sql += " id=" + zeilen[i].substring(zeilen[i].indexOf("[[")+2,zeilen[i].indexOf("]]")) + " OR ";
	         }
	     }
         if(sql.endsWith(" OR ")) {
			sql = sql.substring(0,sql.length() - " OR ".length());
		}
         sql += " ORDER BY extrakt asc";
         ResultSet rs;
         try {
              rs = getReadOnly.executeQuery(sql);
              while (rs.next()) {
                  final RezepturItem r1 = dbGetRezeptur(rs.getInt("id"),true);
                  if(r1.getExtrakt() > 0) {
					cmb.addItem(r1);
				}
              }
         } catch (final Exception e) {
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.setExtraktCombo:", e.getLocalizedMessage());
         }
	 }


    /** füllt eine Combobox mit deb alkoholhaltigen  Zutaten einer Rezeptur
     *
     * @param cmb
     * @param buchung_id
     */
	 public void setAlkoholCmb(JComboBox cmb,String vorgabe,RezepturItem ri){
	     int i;
	     cmb.removeAllItems();
	     final String zeilen[] = ri.getRaw().split("\\n");
	     String sql = "SELECT DISTINCT id,staerke FROM " + DBFactory.TABLE_REZEPTLISTE + " WHERE ";
	     cmb.addItem((Object)dbGetRezeptur(REZEPTURWASSER,true));
	     if(!vorgabe.equals("")) {
	         final String keys[] = vorgabe.split(";");
	         for(i=0;i < keys.length;i++) {
	             sql += " id=" + keys[i] + " OR ";
	     	}
	     }
	     for( i = 0 ; i <zeilen.length;i++){
	         if(zeilen[i].startsWith("1 ")) {
	             	 sql += " id=" + zeilen[i].substring(zeilen[i].indexOf("[[")+2,zeilen[i].indexOf("]]")) + " OR ";
	         }
	     }
         if(sql.endsWith(" OR ")) {
			sql = sql.substring(0,sql.length() - " OR ".length());
		}
         sql += " ORDER BY staerke asc";
         ResultSet rs;
         try {
              rs = getReadOnly.executeQuery(sql);
              while (rs.next()) {
                  final RezepturItem r1 = dbGetRezeptur(rs.getInt("id"),true);
                  if(r1.getStaerke() > 0) {
					cmb.addItem(r1);
				}
              }
              if(rs != null) {
				rs.close();
			}
         } catch (final Exception e) {
             rz.getLogFactory().logMessage(LogFactory.LOG_WARN, "DBRezeptur.setAlkoholCombo:", e.getLocalizedMessage());
         }
	 }
}
