/*
 *  Copyright (C) Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package de.lunqual.rzpro.items.code128;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.image.BufferedImage;


/**
 * @author labor
 * Code 128A|B|C - Modul.
 * erzeugt Code-128 Barcodes aus einem String als Image.
 * Kann Barcode in beliebiger Länge, in einer beliebigen Höhe erzeugen und bei Bedarf
 * den Klartext unter den Code ausgeben.
 * Normalerweise wird man aber das Bild mit der Höhe 1 Pixel ohne Klarext erzeugen und
 * im Reportgenerator auf die gewünschte Höhe skalieren
 *
 */
public class Code128Item {


        int[][]         charCodes = {
                {2,1,2,2,2,2}, //0
                {2,2,2,1,2,2},
                {2,2,2,2,2,1},
                {1,2,1,2,2,3},
                {1,2,1,3,2,2},
                {1,3,1,2,2,2},
                {1,2,2,2,1,3},
                {1,2,2,3,1,2},
                {1,3,2,2,1,2,},
                {2,2,1,2,1,3},//9
                {2,2,1,3,1,2},
                {2,3,1,2,1,2},
                {1,1,2,2,3,2},
                {1,2,2,1,3,2},
                {1,2,2,2,3,1},
                {1,1,3,2,2,2},
                {1,2,3,1,2,2},
                {1,2,3,2,2,1},
                {2,2,3,2,1,1},
                {2,2,1,1,3,2},//19
                {2,2,1,2,3,1},
                {2,1,3,2,1,2},
                {2,2,3,1,1,2},
                {3,1,2,1,3,1},
                {3,1,1,2,2,2},
                {3,2,1,1,2,2},
                {3,2,1,2,2,1},
                {3,1,2,2,1,2},
                {3,2,2,1,1,2},
                {3,2,2,2,1,1},//29
                {2,1,2,1,2,3},
                {2,1,2,3,2,1},
                {2,3,2,1,2,1},
                {1,1,1,3,2,3},
                {1,3,1,1,2,3},
                {1,3,1,3,2,1},
                {1,1,2,3,1,3},
                {1,3,2,1,1,3},
                {1,3,2,3,1,1},
                {2,1,1,3,1,3},//39
                {2,3,1,1,1,3},
                {2,3,1,3,1,1},
                {1,1,2,1,3,3},
                {1,1,2,3,3,1},
                {1,3,2,1,3,1},
                {1,1,3,1,2,3},
                {1,1,3,3,2,1},
                {1,3,3,1,2,1},
                {3,1,3,1,2,1},
                {2,1,1,3,3,1},//49
                {2,3,1,1,3,1},
                {2,1,3,1,1,3},
                {2,1,3,3,1,1},
                {2,1,3,1,3,1},
                {3,1,1,1,2,3},
                {3,1,1,3,2,1},
                {3,3,1,1,2,1},
                {3,1,2,1,1,3},
                {3,1,2,3,1,1},
                {3,3,2,1,1,1},//59
                {3,1,4,1,1,1},
                {2,2,1,4,1,1},
                {4,3,1,1,1,1},
                {1,1,1,2,2,4},
                {1,1,1,4,2,2},
                {1,2,1,1,2,4},
                {1,2,1,4,2,1},
                {1,4,1,1,2,2},
                {1,4,1,2,2,1},
                {1,1,2,2,1,4},//69
                {1,1,2,4,1,2},
                {1,2,2,1,1,4},
                {1,2,2,4,1,1},
                {1,4,2,1,1,2},
                {1,4,2,2,1,1},
                {2,4,1,2,1,1},
                {2,2,1,1,1,4},
                {4,1,3,1,1,1},
                {2,4,1,1,1,2},
                {1,3,4,1,1,1},//79
                {1,1,1,2,4,2},//80
                {1,2,1,1,4,2},//81
                {1,2,1,2,4,1},//82
                {1,1,4,2,1,2},//83
                {1,2,4,1,1,2},//84
                {1,2,4,2,1,1},//85
                {4,1,1,2,1,2},//86
                {4,2,1,1,1,2},
                {4,2,1,2,1,1},
                {2,1,2,1,4,1},//89
                {2,1,4,1,2,1},
                {4,1,2,1,2,1},
                {1,1,1,1,4,3},
                {1,1,1,3,4,1},
                {1,3,1,1,4,1},
                {1,1,4,1,1,3},
                {1,1,4,3,1,1},
                {4,1,1,1,1,3},
                {4,1,1,3,1,1},
                {1,1,3,1,4,1},//99
                {1,1,1,1,3,1},
                {3,1,1,1,4,1},
                {4,1,1,1,3,1},
                {2,1,1,4,1,2},
                {2,1,1,2,1,4},
                {2,1,1,2,3,2},
        };
        int [] stopCode = {2,3,3,1,1,1,2};


        private static final int       	BAR_WIDTH   = 10;
        private static final double	HEIGHT_FAKTOR = 0.3;
        public static final int			CODE_A	= 103;
        public static final int			CODE_B	= 104;
        public static final int			CODE_C	= 105;

        public Code128Item() {
        }
        /**
         * gibt ein Image zurück, welches den Barcode enthält
         * @param type 	CODE_B|CODE_B|CODE_C
         * @param wantedHeight  Höhe/Breite-Verhältnis oder automatische Berechnung der Höhe
         * 										wenn <0, dann ist das Image nur 1 Pixel hoch und muss später skaliert werden
         * 										wenn == 0, dann ist das Image HEIGHT_FAKTOR der Breite hoch
         * 										wenn > 0 dann ist das Image wantedHeight % der Breite hoch
         * @param verbose			wenn false==nur Barcode, wenn true==wird der Code noch lesbar
         * 										darunter gedruckt
         * @param code				der String, der codiert werden soll
         * @return	Image, das den Barcode und ggfs. den ausgeschriebenen Code dazu enthält
         */
        public Image get128Image(int type,int wantedHeight,boolean verbose,String code) {
        	//code = clearString(code);
            Graphics2D graphics = null;
            Image img = null;
            int height = 1;
            final int width = (code.length() + 5) * BAR_WIDTH * 11;
            if(wantedHeight < 0) {
            	height = 1;
            } else if (wantedHeight == 0) {
            	height = (int)(width * HEIGHT_FAKTOR);
            } else {
            	height = (width * wantedHeight);
            }
            img  = new BufferedImage(width,height,BufferedImage.TYPE_BYTE_BINARY);
            graphics = (Graphics2D)img.getGraphics();
            graphics.setClip(0,0, width, height);
            graphics.setColor(new Color(255,255,255));
            graphics.fillRect(0,0,(int)graphics.getClipBounds().getWidth(),(int)graphics.getClipBounds().getHeight());
            paint128(graphics,type,code);
            if(verbose) {
            	paintCode(graphics,code,width,height);
            }
            return img;
        }

        private void paint128(Graphics2D graphics,int type,String code) {
        	//code = clearString(code);
            final int codes[] = new int[code.length() + 3];
            int summe = type;
            int i;
            codes[0] = type;
            codes[codes.length-1]=106;
            for( i = 0;i < code.length();i++) {
            	if(code.substring(i).equals("ä"))
                codes[i+1] = (byte)(code.charAt(i)-32);
                if( codes[i+1] > 106) {
                    codes[i+1] = 0;
                }
                summe += (i+1)*codes[i+1];
            }
            codes[codes.length-2]=summe % 103;
            int startx = 10 * BAR_WIDTH;
            for(i=0;i < codes.length;i++) {
                paintSingle(graphics,startx,codes[i]);
                startx += BAR_WIDTH * 11;
            }
        }

        private String clearString(String code) {
        	return code.replaceAll("ä","a")
        			.replaceAll("Ä","A")
        			.replaceAll("ü", "u")
        			.replaceAll("Ü","U")
        			.replaceAll("ö", "o")
        			.replaceAll("Ö","O");
        			
        }
        
        private void paintSingle(Graphics2D graphics,int x,int code) {
            int x1 = x;
            final Color black = new Color(0,0,0);
            final Color white = new Color(255,255,255);
            final Color[] colors = {black,white,black,white,black,white,black};
            final int[] bars = (code==106) ? stopCode: charCodes[code];
            for(int j=0;j< bars.length;j++) {
                graphics.setColor(colors[j]);
                graphics.fillRect(x1, 0, bars[j] * BAR_WIDTH, (int)graphics.getClipBounds().getHeight());
                x1+=bars[j] * BAR_WIDTH;
            }
        }

        private void paintCode(Graphics2D graphics,String code,int width,int height) {
            final Font fnt = new Font("Default",0,(height/5));
            graphics.setFont(fnt);
            graphics.setColor(new Color(255,255,255));
            graphics.fillRect(0,height-(height/5),(int)graphics.getClipBounds().getWidth(),height/5);
            graphics.setColor(new Color(0,0,0));
            graphics.drawString(code, BAR_WIDTH * 10, height);
        }
}
