/*
 *  Copyright (C) 2004 Karlheinz Klingbeil (lunqual)
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MeRCHANTABILITY or FITNeSS FOR A PARTICULAR PURPOSe.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package de.lunqual.rzpro.tafel;

import java.math.BigDecimal;
import java.text.NumberFormat;
import java.text.ParseException;

import de.lunqual.rzpro.RzPro;
import de.lunqual.rzpro.options.OptionFactory;

/**
 *
 * @author  lunqual
 */
public class TafelFactory {
	RzPro								rz;
    double rhoWasser       = 0.99820123;
    public double rhoAlk          = 0.789239;
    double nullPrecision = 0.0001;
    double brennWertAlkohol = 29; // Brennwert des Alkohols in Kj/Gramm
    boolean	genau = false;

    double[] co = {998.20123,-192.9769495,389.1238958,-1668.103923,13522.15441,
       -88292.78388,306287.4042,-613838.1234,747017.2998, -547846.1354,223446.0334,-39032.85426 };

    double[] cb = {-2.0618513e-1,-5.2682542e-3,3.6130013e-5,-3.8957702e-7,
        7.169354e-9,-9.9739231e-11 };

    double [][] ck =
    {
        { 1.693443461530087e-1,-1.046914743455169e+1,7.196353469546523e+1,
         -7.047478054272792e+2,3.924090430035045e+3,-1.210164659068747e+4,
         2.248646550400788e+4,-2.605562982188164e+4,1.852373922069467e+4,
        -7.420201433430137e+3,1.285617841998974e+3
        },
        {
         -1.19301300505701e-2,2.517399633803461e-1,-2.170575700563993,
         1.353034988843029e+1,-5.029988758537014e+1,1.09635566657757e+2,
         -1.422753946421155e+2,1.08043594285623e+2,-4.414153236817392e+1,
         7.442971530188783,0
        },
        {
         -6.802995733503803e-4,1.876837790289664e-2,-0.2002561813734156,
         1.02299296671922,-2.895696483903638,4.810060584300675,
         -4.672147440794683,2.458043105903461,-5.411227621436812e-1,
         0,0
        },
        {
         4.075376675622027e-6,-8.76305857347111e-6,6.515031360099368e-6,
         -1.51578483698721e-6,0,0,0,0,0,0,0
        },
        {
         -2.788074354782409e-08,1.345612883493354e-08,
         0,0,0,0,0,0,0,0,0
        }
    };

    int [] ci =
    {
      11,10,9,4,2
    };

    private double[] flammpunkte = {
    		45,		//14
    		44,42.5,	42,41,40,
    		39,//20
    		38,37,	36,35.5,
    		34.5,//25
    		34,33,32.5,32,
    		31.5,//30
    		31,30.5,30,29.5,
    		29,//35
    		28.5,28,27.6,27.5,
    		27.2,//40
    		27,26.5,26.4,26.2,
    		26,//45
    		25.7,25.5,25.3,25.1,
    		24.8,//50
    		24.7,24.6,24.5,24.4,
    		24.3,//55
    		24,23.8,23.7,23.6,
    		23.5,//60
    		23.4,23.3,23.0,22.8,
    		22.7,//65
    		22.6,22.5,22.3,22.2,
    		22,//70
    		21.9,21.8,21.7,21.6,
    		21.5,//75
    		21.2,21,20.8,20.5,
    		20.3,//80
    		20.1,20,19.8,19.6,
    		19.3,//85
    		19,18.8,18.7,18.3,
    		18.1,//90
    		17.8,17.5,17,16.7,
    		16.5,//95
    		16.2,//96
    		15.6
    };
    
    NumberFormat nf;
    /** Creates a new instance of tafel */
    public TafelFactory(RzPro rz) {
    	this.rz = rz;
        nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(1);

    }

    private double getFlammpunktValue(double value) {
    	double ret = 45;
    	try {
    		int index = (int)(Math.floor(value));
    		double decimal = value - (double)index;
    		index = index-14;
    		if(!rz.isZero(decimal)) {
        		double span  = flammpunkte[index] - flammpunkte[index+1];
    			ret = flammpunkte[index] - span * decimal;
    		} else {
    			ret = flammpunkte[index];
    		}
    	}catch(Exception e) {}
    	return ret;
    }
    
    public String getFlammpunkt(double value) {
    	String ret = rz.getLocale().getString("tafel.flammpunkt.text");
    	if(value<14) {
    		ret = ret.replaceAll("%p", ">");
    		ret = ret.replaceAll("%f", rz.getLocale().formatNumber(flammpunkte[0],OptionFactory.NF_ONE));
    	} else if(value > 96) {
    		ret = ret.replaceAll("%p", "<");
    		ret = ret.replaceAll("%f", rz.getLocale().formatNumber(flammpunkte[flammpunkte.length-1],OptionFactory.NF_ONE));
    	} else {
    		ret = ret.replaceAll("%p", " ");
    		ret = ret.replaceAll("%f", rz.getLocale().formatNumber(getFlammpunktValue(value),OptionFactory.NF_NORMAL));
    	}
    	return ret;
    }
    
    public double dt6(double MasProz){
        double m;
        double rho;
        int i;
        m = MasProz / 100;
        rho = co[0];
        for(i=1;i < co.length;i++){
           rho = rho + co[i] * java.lang.Math.pow(m,i);
        }
        rho = rho/1000;
        return rho;
    }


    public double MasProz2VolProz(double MasProz){
        return (MasProz * dt6(MasProz)) / rhoAlk;
    }

    public double VolProz2MasProz(double VolProz){
        double alt;
        double neu;
        neu = VolProz * rhoAlk / dt6(VolProz);
        do {
          alt = neu;
          neu = VolProz * rhoAlk / dt6(alt);
        }
        while ((alt - neu) > 0.001);
        return neu;
    }


    public double Dichte(double VolProz , double Temperatur){
        double  mas;
        int     bi;
        int     i;
        int     k;
        double  sb;
        double  sc;
        double  ret;
        sb = 0.0;
        sc = 0.0;

        final double t1 = Temperatur - 20;
        mas = VolProz2MasProz(VolProz);
        final double m1 = mas/100;

        for(bi = 0 ;bi < cb.length; bi++){
          sb += (cb[bi] * java.lang.Math.pow(Temperatur - 20, (bi+1)));
        }
	      for( i =0;i < ci.length;i++){
	        for (k = 0; k < ci[i];k++){
	          sc += (ck[i][k] * java.lang.Math.pow(m1,k+1) * java.lang.Math.pow(t1, i+1));
	        }
	      }

        ret = dt6(mas)+ ((sb + sc) / 1000);
        return ret;
    }

    public double Tafel2(double VolProz){
        double v;
        double ret;
        v = dt6(VolProz2MasProz(VolProz)) * 1000;
        ret = (100 * VolProz * (1000 / v) * (1 + (1.2 / v) - 0.00015))/10;
        return ret;
    }

    public double Tafel2Tafel(double VolProz){
        double v;
        double ret;
        double x;
        v = dt6(VolProz2MasProz(VolProz)) * 1000;
        x = (100 * VolProz * (1000 / v) * (1 + (1.2 / v) - 0.00015))/10;
        try {
            ret = nf.parse(nf.format(x)).doubleValue();
        } catch(final ParseException e) {
            ret = x;
        }
        return ret;
    }
    /**
     * Tafel 2 nach der amtlichen Alkoholtafel gerundet
     * @param VolProz
     * @param Temperatur
     * @return
     */
    public double Tafel2Tafel(double VolProz,double Temperatur){
        double v;
        double ret;
        final double vt = Dichte(VolProz, Temperatur) * 1000;
        v = dt6(VolProz2MasProz(VolProz)) * 1000;
        final double x = ((100 * VolProz * (1000 / v) * (1 + (1.2 / vt) - 0.00015)) ) / 10;
        try {
            ret = nf.parse(nf.format(x)).doubleValue();
        } catch(final ParseException e) {
            ret = x;
        }
        return ret;
    }

    /**
     * Tafel 2 genau ohne Rundung
     * @param VolProz
     * @param Temperatur
     * @return
     */
    public double Tafel2(double VolProz,double Temperatur){
        double v;
        double ret;
        final double vt = Dichte(VolProz, Temperatur) * 1000;
        v = dt6(VolProz2MasProz(VolProz)) * 1000;
        ret = ((100 * VolProz * (1000 / v) * (1 + (1.2 / vt) - 0.00015)) ) / 10;
        return ret;
    }

    public double Tafel3_b(double VolProz, double Temperatur){
        double dichte_20;
        double dichte_t;
        double ret;
        dichte_20 = dt6(VolProz2MasProz(VolProz));
        dichte_t  = Dichte(VolProz, Temperatur);
        ret = (VolProz / 100) * (dichte_t / dichte_20) * (1000 * (1 + (0.000036 * (Temperatur - 20))));
        return ret;
    }
    /**
     * Tafel 3 nach der amtlichen Alkoholtafel gerundet
     * @param VolProz
     * @param Temperatur
     * @return tafel3
     */
    public double Tafel3Tafel(double VolProz, double Temperatur ){
    	final BigDecimal oben = new BigDecimal(Tafel3_b(java.lang.Math.floor(VolProz + 1), Temperatur) ).setScale(1,BigDecimal.ROUND_HALF_UP);
    	final BigDecimal unten = new BigDecimal(Tafel3_b(java.lang.Math.floor(VolProz), Temperatur )).setScale(1,BigDecimal.ROUND_HALF_UP);
    	final BigDecimal diff = new BigDecimal(VolProz - java.lang.Math.floor(VolProz)).setScale(2,BigDecimal.ROUND_HALF_UP);
    	final BigDecimal bd = (oben.subtract(unten).multiply( diff).add(unten)).setScale(2,BigDecimal.ROUND_HALF_UP);

        return bd.doubleValue();
    }

    public double Tafel3(double VolProz, double Temperatur ){
        return  ((Tafel3_b(java.lang.Math.floor(VolProz + 1), Temperatur) - Tafel3_b(java.lang.Math.floor(VolProz), Temperatur)) * (VolProz - java.lang.Math.floor(VolProz))) + Tafel3_b(java.lang.Math.floor(VolProz), Temperatur);
    }

    public double Tafel4(double VolProz){
        return VolProz2MasProz(VolProz);
    }

    public double Tafel5(double MasProz){
     return MasProz2VolProz(MasProz);
    }

    public double Staerke(double liter,double la) {
        if(liter != 0) {
            return (100 * la)/liter;
        } else {
            return 0;
        }
    }

    public double LitergewichtTafel(double VolProz) {
        double ret;
        if ((VolProz > 0) && (VolProz <= 100)){
          ret = (1000 / Tafel2Tafel(VolProz)) * (VolProz / 100);
        }
        else {
          ret = rhoWasser;
        }
        return ret;
    }

    public double LitergewichtTafel(double VolProz,double Temperatur) {
        double ret;
        if ((VolProz > 0) && (VolProz <= 100)){
          final double f3 = Tafel3(VolProz, Temperatur);
          ret= (1000 /Tafel2Tafel(VolProz, Temperatur)) * (f3 / 1000);
        }
        else {
          ret = rhoWasser;
        }
        return ret;
    }

    public double Litergewicht(double VolProz,double Temperatur){
        double ret;
        if ((VolProz > 0) && (VolProz <= 100)){
          final double f3 = Tafel3(VolProz, Temperatur);
          ret= (1000 /Tafel2(VolProz, Temperatur)) * (f3 / 1000);
        }
        else {
          ret = rhoWasser;
        }
        return ret;
    }

    public double Litergewicht(double VolProz){
        double ret;
        if ((VolProz > 0) && (VolProz <= 100)){
          ret = (1000 / Tafel2(VolProz)) * (VolProz / 100);
        }
        else {
          ret = rhoWasser;
        }
        return ret;
    }

    /**
     * Brennwert in Kj/100 ml
     * @param staerke in VolProz
     * @return Brennwert in Kj/100ml
     *
     */
    public double Brennwert(double staerke) {
    	return  staerke * Litergewicht(100) * this.brennWertAlkohol;
    }

}